C***********************************************************************
C                 copyright 2001, Amoco Production Company             *
C                             All Rights Reserved                      *
C                     an affiliate of BP America Inc.                  *
C***********************************************************************
C**********************************************************************C
C**********************************************************************C
C
C      PROGRAM MODULE RNMOSPEC: residual nmo vel spectra (as sis files)
C
C**********************************************************************C
c      B.V. NGUYEN and R.B. HERRMANN                     APRIL 1986 SLU
C**********************************************************************C
C          THE MAIN PROGRAM CALL SUBROUTINE TVEL TO PERFORM HYPERBOLIC-
C      TRAJECTORY SCAN SEARCH FOR OPTIMUM STACKING RNMO (RMS VEL.)
C      AS A FUNCTION OF TWO-WAY ZERO-OFFSET TIME VIA A COHERENCY MEASURE
C      CALLED SEMBLANCE IN A CMP GATHER OF TRACES.
C      THE SURFACE REPRESENTING THE COHERENCY MEASURE AS A FUNCTION OF
C      TWO-WAY ZERO-OFFSET TIME AND TEST STACKING RNMO DEFINES A
C      RNMO SPECTRUM.
C      FOR SEMBLANCE REFERENCE:
C      TANER, M.T. and F. KOEHLER (1969) GEOPHYSICS,34,p.859-881.
C      FOR ENERGY-NORMALIZED CROSS-CORRELATION SUM REFERENCE:
C      HUBRAL, P. and T. KREY (LARNER, K. L., Ed.) (1980) SOCIETY OF
C      EXPLORATION GEOPHYSICISTS, 203pp.
C**********************************************************************C

#include <localsys.h>
#include <save_defs.h>
#include <f77/iounit.h>
#include <f77/lhdrsz.h>
#include <f77/sisdef.h>

      INTEGER     ITR ( SZLNHD )
      INTEGER     STR ( SZLNHD )
      INTEGER     LHED( SZLNHD ), LUIN, NBYTES, OBYTES
      INTEGER     NSAMP, NSI, NTRC, NREC, IFORM, LBYTES
      INTEGER     ISTOP, ISTART, NPT
      REAL        START,STOP
      REAL        TRI ( SZLNHD )
      CHARACTER   NAME * 8, ntap * 256, otap * 256
#include <f77/pid.h>
      logical     verbos
      logical     first
      logical     abslut
      logical     query, heap
      integer     argis
c     real        traces (SZSMPM,SZSPRD)
      real        traces
      pointer     (wkaddr, traces(1))
      real        tmul, dmul
      real        dis(SZLNHD)
      integer     nvel, irec
      integer     ns,ne
      real        vmin,vmax
      integer     igt, pipe
      integer     errcod, abort

c     EQUIVALENCE ( ITR(129), TRI (1) )
c     EQUIVALENCE ( VTR(129), VRI (1) )
      EQUIVALENCE ( ITR(  1), LHED(1) )
      DATA NAME /'RNMOSPEC'/, NBYTES / 0 /, LBYTES / 0 /
      DATA pipe/3/
      data abort / 0 /
      data first/.true./

C*********************************************************************C
c  get online help if necessary
C*********************************************************************C
      query = ( argis ('-?').gt.0 .or. argis('-h').gt.0 )
      if ( query ) then
           call help ()
           stop
      endif

C*********************************************************************C
c  open printout file
C*********************************************************************C
#include <f77/open.h>

C**********************************************************************C
C     READ PROGRAM PARAMETERS FROM COMMAND LINE
C**********************************************************************C
      call cmdln (ntap,otap,start,stop,dmul,tmul,igt,xmin,xmax,
     &               ns,ne,irs,ire,nvel,vmin,vmax,abslut,verbos,
     &               iscl)

C**********************************************************************C
C     READ AND UPDATE LINE HEADER,
C     WRITE LINE HEADER, SAVE KEY PARAMETERS.
C**********************************************************************C
      call getln ( luin, ntap, 'r', 0 )
      call getln (luout, otap, 'w', 1 )


      lbytes = 0
      CALL RTAPE ( LUIN, itr, LBYTES           )
      if(lbytes .eq. 0) then
         write(LERR,*)'RNMOSPEC: no header read on unit ',ntap
         write(LERR,*)'FATAL'
         write(LERR,*)'Check existence of file & rerun'
         stop
      endif
      CALL HLHprt ( ITR , LBYTES, name, 8, LERR         )

c-----------------------------------------
c  save key header values
#include <f77/saveh.h>

c     For trace header values we take mnemonics and build a
c     set of pointers to an I*2 array equivalenced to the
c     RTAPE  Integer array (headers + data)
c     TRACEHEADER is a value in the include file <sisdef.h> that
c     refers to the trace header
 
      call savelu('TrcNum',ifmt_TrcNum,l_TrcNum,ln_TrcNum,TRACEHEADER)
      call savelu('RecNum',ifmt_RecNum,l_RecNum,ln_RecNum,TRACEHEADER)
      call savelu('SrcLoc',ifmt_SrcLoc,l_SrcLoc,ln_SrcLoc,TRACEHEADER)
      call savelu('RecInd',ifmt_RecInd,l_RecInd,ln_RecInd,TRACEHEADER)
      call savelu('DphInd',ifmt_DphInd,l_DphInd,ln_DphInd,TRACEHEADER)
      call savelu('LinInd',ifmt_LinInd,l_LinInd,ln_LinInd,TRACEHEADER)
      call savelu('DstSgn',ifmt_DstSgn,l_DstSgn,ln_DstSgn,TRACEHEADER)
      call savelu('DstUsg',ifmt_DstUsg,l_DstUsg,ln_DstUsg,TRACEHEADER)
      call savelu('StaCor',ifmt_StaCor,l_StaCor,ln_StaCor,TRACEHEADER)
      call savelu('SrPtXC',ifmt_SrPtXC,l_SrPtXC,ln_SrPtXC,TRACEHEADER)
      call savelu('SrPtYC',ifmt_SrPtYC,l_SrPtYC,ln_SrPtYC,TRACEHEADER)
      call savelu('RcPtXC',ifmt_RcPtXC,l_RcPtXC,ln_RcPtXC,TRACEHEADER)
      call savelu('RcPtYC',ifmt_RcPtYC,l_RcPtYC,ln_RcPtYC,TRACEHEADER)
      call savelu('CDPBCX',ifmt_CDPBCX,l_CDPBCX,ln_CDPBCX,TRACEHEADER)
      call savelu('CDPBCY',ifmt_CDPBCY,l_CDPBCY,ln_CDPBCY,TRACEHEADER)


c     if ( verbos ) then
         write(LERR,*)' '
         write(LERR,*)' Values read from input data set lineheader'
         write(LERR,*)' # of Samples/Trace =  ',nsamp
         write(LERR,*)' Sample Interval    =  ',nsi
         write(LERR,*)' Traces per Record  =  ',ntrc
         write(LERR,*)' Records per Line   =  ',nrec
         write(LERR,*)' Format of Data     =  ',iform
c     end if

C**********************************************************************C
C     CHECK DEFAULT VALUES AND SET PARAMETERS
C**********************************************************************C
      dvel  = (vmax-vmin)/(NVEL-1)
      if(START.le.0.0)START = 0.0
      ttmax = (NSAMP-1)*NSI
      if(STOP.le.0.0 .or. STOP.ge. ttmax)STOP=ttmax
      dt=real(NSI)
      dt=dt*tmul
      ISTART =   INT   ( START / dt ) + 1
      ISTOP  =   INT   ( STOP  / dt ) + 1
        IF ( ISTART . LT. 1     .OR. ISTART.GT. NSAMP ) THEN
           ISTART = 1
        ENDIF
        IF ( ISTOP . LE. ISTART .OR. ISTOP .GT. NSAMP ) THEN
           ISTOP  = NSAMP
        ENDIF
      NPT    =       ISTOP - ISTART + 1
      if(ire.lt.1.or.ire.lt.irs.or.ire.gt.nrec)ire = nrec
      if(ns.le.0 )ns = 1
      if(ne.le.0 .or. ne.gt.ntrc)ne=ntrc

C**********************************************************************C
c     if ( verbos ) then
         write(LERR,*)' Values read from file command line'
         write(LERR,*)' Start time in ms (default = 0 ms)    =  ',START
         write(LERR,*)' Stop time in ms (default= full trace) =  ',STOP
         write(LERR,*)' Number of RNMO inputs             =  ',nvel
         write(LERR,*)' Minimum RNMO for plots            =  ',vmin
         write(LERR,*)' Maximum RNMO for plots            =  ',vmax
         write(LERR,*)' Delta RNMO                        =  ',dvel
         write(LERR,*)' Gamma scale factor                =  ',iscl
         write(LERR,*)' Time gate integer                     =  ',igt
         write(LERR,*)' Distance multiplied by a factor       =  ',dmul
         write(LERR,*)' Time multiplied by a factor           =  ',tmul
         write(LERR,*)' First record to process               =  ',irs
         write(LERR,*)' Last record to process                =  ',ire
         write(LERR,*)' Starting trace number                 =  ',ns
         write(LERR,*)' Ending   trace number                 =  ',ne
c     end if
C**********************************************************************C


c---------------------------------------------------
c  malloc only space we're going to use
      heap = .true.

c - switch from machine-dependent to generic memory allocation
c - j.m.wade 8/20/92

      items = ntrc * (nsamp)
      call galloc (wkaddr, items*SZSMPD, errcod, abort)
      if (errcod .ne. 0) heap = .false.

      if (.not. heap) then
         write(LERR,*)' '
         write(LERR,*)'Unable to allocate workspace:'
         write(LERR,*) items*SZSMPD,'  bytes'
         write(LERR,*)' '
         go to 999
      else
         write(LERR,*)' '
         write(LERR,*)'Allocating workspace:'
         write(LERR,*) items*SZHFWD,'  bytes'
         write(LERR,*)' '
      endif
c---------------------------------------------------
c
      scl = iscl
      ivmin = vmin
      ivmax = vmax
c--------------------------------------------------------
c  adjust line header, historical l.h., & output header
       call savew( itr, 'NumTrc', nvel , LINHED)
       call savew( itr, 'NumSmp', istop, LINHED)
       call savew( itr, 'T_Unit', iscl , LINHED)
       call savew( itr, 'Crew01', 'G'  , LINHED)
       call savew( itr, 'Crew02', 'a'  , LINHED)
       call savew( itr, 'Crew03', 'm'  , LINHED)
       call savew( itr, 'Crew04', 'm'  , LINHED)
       call savew( itr, 'Crew05', 'a'  , LINHED)
       call savew( itr, 'MinVel', ivmin, LINHED)
       call savew( itr, 'MaxVel', ivmax, LINHED)
       call savew( itr, 'ILClIn',  vmin, LINHED)
       call savew( itr, 'CLClIn',  dvel, LINHED)
       obytes = SZSMPD * (istop + ITRWRD)
       call savhlh ( itr, lbytes, lbyout )
       call wrtape(luout, itr, lbyout )

c---------------------------------------
c  skip to start record
      call recskp(1,irs-1,luin,ntrc,itr)

C**********************************************************************C
C
C      READ TRACES, COMPUTE SPECTRA, WRITE RESULTS
C
C**********************************************************************C
        last = 0
        ist=0
        DO 5001 JJ = irs, ire

c-----------------------------------------------
c  skip to start trace of current record
            call trcskp(jj,1,ns-1,luin,ntrc,itr)

           ist=ist+1
           k=0
c----------------------------
c  read selected data

           valmx = 0.
           valmy = 0.
           idtot = 0
           iltot = 0

           DO 5002 KK = ns, ne

                 NBYTES = 0
                 CALL RTAPE  ( LUIN, ITR, NBYTES )
                 if(nbytes .eq. 0) then
                    write(LERR,*)'End of file on input:'
                    write(LERR,*)'  rec= ',jj,'  trace= ',kk
                    go to 999
                 endif
                 call vmov (itr(ITHWP1), 1, tri, 1, nsamp)

                 call saver2(itr,ifmt_StaCor,l_StaCor, ln_StaCor,
     1                       istatic , TRACEHEADER)
                 call saver2(itr,ifmt_DstSgn,l_DstSgn, ln_DstSgn,
     1                        idist , TRACEHEADER)
                 call saver2(itr,ifmt_RecNum,l_RecNum, ln_RecNum,
     1                       irec , TRACEHEADER)
                 call saver2(itr,ifmt_DphInd,l_DphInd, ln_DphInd,
     1                         idi  , TRACEHEADER)
                 call saver2(itr,ifmt_LinInd,l_LinInd, ln_LinInd,
     1                         ili  , TRACEHEADER)
                 call saver2(itr,ifmt_SrPtXC,l_SrPtXC, ln_SrPtXC,
     1                       ivalsx  , TRACEHEADER)
                 call saver2(itr,ifmt_SrPtYC,l_SrPtYC, ln_SrPtYC,
     1                       ivalsy  , TRACEHEADER)
                 call saver2(itr,ifmt_RcPtXC,l_RcPtXC, ln_RcPtXC,
     1                       ivalrx  , TRACEHEADER)
                 call saver2(itr,ifmt_RcPtYC,l_RcPtYC, ln_RcPtYC,
     1                       ivalry  , TRACEHEADER)
                 idtot = idtot + idi
                 iltot = iltot + ili
                 valmx = valmx + .5 * (float (ivalsx+ivalrx)) + 0.5
                 valmy = valmy + .5 * (float (ivalsy+ivalry)) + 0.5

                 IF (istatic .ne. 30000) THEN
                     if (kk .eq. ns)
     1               call vmov (itr, 1, str, 1, ITRWRD)
                     k=k+1
                     dis(k) = iabs(idist) * dmul
                     istrc  = (k-1) * nsamp
                     do 5004 LL= 1, nsamp
                           traces(istrc + LL) = tri(LL)
 5004                continue
                 ENDIF

 5002      CONTINUE

           if (k .gt. 0) then
                 idtot = idtot / k
                 iltot = iltot / k
                 valmx = valmx / float(k)
                 valmy = valmy / float(k)
           endif

c----------------------------------
c  do spectra
      call tvel(traces,str,tri,dt,nvel,k,dis,scl,unitsc,
     1        vmin,vmax,luout,obytes,xmin,xmax,irec,
     2        ntap,JJ,ns,ne,NSI,NSAMP,ISTART,ISTOP,START,STOP,igt,
     3          abslut,irs,ire,ist,ITRWRD,ITHWP1,SZLNHD,
     4          ifmt_RecNum,l_RecNum, ln_RecNum,
     5          ifmt_DstSgn,l_DstSgn, ln_DstSgn,
     6          ifmt_TrcNum,l_TrcNum, ln_TrcNum,
     7          ifmt_DphInd,l_DphInd,ln_DphInd,
     8          ifmt_LinInd,l_LinInd,ln_LinInd,
     9          ifmt_StaCor,l_StaCor,ln_StaCor,
     1          ifmt_CDPBCX,l_CDPBCX,ln_CDPBCX,
     2          ifmt_CDPBCY,l_CDPBCY,ln_CDPBCY,
     4          idtot,iltot,valmx,valmy,first)

      write(LER,*)'Completed Record= ',JJ
c-----------------------------------------------
c  skip to end trace of current record
            call trcskp(jj,ne+1,ntrc,luin,ntrc,itr)

 5001 CONTINUE

  999 continue
      call lbclos( luin)
      call lbclos(luout)
      END
C**********************************************************************C

C**********************************************************************C
      subroutine tvel(traces,itr,tri,dt,nvel,k,dis,scl,unitsc,
     1        vmin,vmax,luout,obytes,xmin,xmax,irec,
     2          ntap,nrec,ktrc1,ktrc2,nsi,nsamp,ISTART,ISTOP,START,
     3          STOP,igt,abslut,irs,ire,ist,ITRWRD,ITHWP1,SZLNHD,
     4          ifmt_RecNum,l_RecNum, ln_RecNum,
     5          ifmt_DstSgn,l_DstSgn, ln_DstSgn,
     6          ifmt_TrcNum,l_TrcNum, ln_TrcNum,
     7          ifmt_DphInd,l_DphInd,ln_DphInd,
     8          ifmt_LinInd,l_LinInd,ln_LinInd,
     9          ifmt_StaCor,l_StaCor,ln_StaCor,
     1          ifmt_CDPBCX,l_CDPBCX,ln_CDPBCX,
     2          ifmt_CDPBCY,l_CDPBCY,ln_CDPBCY,
     4          idtot,iltot,valmx,valmy,first)

C**********************************************************************C
c      SUBROUTINE ARGUMENTS
c
c      traces    R*4      Two dimensional array of spectra of trace
c                         traces(LL,LR) for trace LR and time LL
c      obytes    I*4      output number of bytes
c      itr      I*4      output data vector equivalenced to...
c...   tri       R*4      ...output real data vector
c      luout     I*4      output data unit number
c      istart    I*4      start time in samples
c      start     R*4      start time in ms
c      istop     I*4      stop time in samples
c      stop      R*4      stop time in ms
c      dt        R*4      Sample interval in MILLISECONDS
c      nvel      I*4      Number of RNMO inputs
c      k         I*4      Number of traces
c      dis       R*4      Array of source-receiver distances
c      vmin      R*4      Minimum RNMO  (ft/sec)for plot
c      vmax      R*4      Maximum RNMO  (ft/sec)for plot
c      ntap      C*120    File name
c      nrec      I*4      Record number
c      ktrc1     I*4      Starting trace number
c      ktrc2     I*4      Ending trace number
c      nsi       I*4      Sample interval in MILLISECONDS
c      NSAMP     I*4      Number of samples in original time series
c      IGT       I*4      integer for time gate centered at zero-offset time
c                         semblance varies between 0 and 1
c      abslut    L        contour absolute semblance rather than relative
C**********************************************************************C
c

#include <f77/iounit.h>
c#include <f77/lhdrsz.h>
#include <f77/sisdef.h>

      real      traces(nsamp,k),tri(*)
      integer   itr(*)
      integer   ITRWRD,ITHWP1,SZLNHD
      integer   IGT,NVEL,IO
      real      dis(*)
      real      cnnum,cnden,xtrc,amp
      real      gi, r
      real      snum(10000),sden(10000)
      real      cn(10000)
      real      xx(2000)
      logical   abslut
      logical   first

      xtrc=k
      si = float(nsi)
      dt1 = dt * unitsc

      do  11  LR = 1, k
          xx(lr) = dis(LR)
          dis(LR) = ( dis(LR) / 2 ) **2
11    continue

C**********************************************************************C
c      perform scan of trajectories to determine vrms
C**********************************************************************C
      recmax = 0.0
      dvel=(vmax-vmin)/(NVEL-1)
      if (first) then
         write(LERR,*)'RNMO start/end gammas = ',vmin,vmax
         write(LERR,*)'RNMO increment (gamma)=  ',dvel,' ngamma= ',nvel
         write(LER ,*)'RNMO start/end gammas = ',vmin,vmax
         write(LER ,*)'RNMO increment (gamma)=  ',dvel,' ngamma= ',nvel
      endif
C**********************************************************************C
C**********************************************************************C
c       for each RNMO, dv,  on either side of V(i)
C**********************************************************************C
      DO 100 IR = 1,NVEL

            do 45 LL=1,ISTOP
               cn(LL)=0.
   45       continue

c--------------
c delta v
                gi = vmin + (IR-1)*dvel
                ig = gi
                gi = gi / scl
                 call savew2(itr,ifmt_DstSgn,l_DstSgn, ln_DstSgn,
     1                        ig , TRACEHEADER)
                 call savew2(itr,ifmt_TrcNum,l_TrcNum, ln_TrcNum,
     1                        IR , TRACEHEADER)

            DO 51 LL=ISTART,ISTOP
              snum(LL)=0.
51            sden(LL)=0.
C**********************************************************************C
      DO 102 LR = 1,k
           r = dis(LR)
C**********************************************************************C
           do  LL = ISTART,ISTOP
                 z0 = float(LL-1) * si
                 tt = z0 * z0 +  (gi * gi - 1) * r
                 if (tt .ge. 0.) then
                    it = sqrt (tt) / si + 1
                    if(it.lt.1)then
                          amp = 0.0
                    elseif(it.ge.nsamp)then
                          amp = 0.0
                    else
                          amp = traces (it,LR)
                    endif
                    snum(LL) = snum(LL) + amp
                    sden(LL) = sden(LL) + amp*amp
                 endif
           enddo
102   CONTINUE
C**********************************************************************C
c      square numerator term
C**********************************************************************C
            DO 103 LL=ISTART,ISTOP
               snum(LL)=snum(LL)**2
103         continue
C**********************************************************************C
c      decide for a time gate
C**********************************************************************C
      IGTP= ISTART + IGT
      IGTM= ISTOP  - IGT
      IO= IGTM - IGTP + 1
C**********************************************************************C
c      begin to compute semblance for a given RNMO
C**********************************************************************C
      DO 200 LL=IGTP,IGTM
            cnnum=0.
            cnden=0.
C**********************************************************************C
            IGL=LL-IGT
            if(IGL.le.0)IGL=1
            IGH=LL+IGT
            if(IGH.gt.ISTOP)IGH=ISTOP
C**********************************************************************C
            DO 305 I=IGL,IGH
              cnnum=cnnum + snum(I)
              cnden=cnden + sden(I)
305         continue
C**********************************************************************C
c
c   put correlation z-values into big matrix
            if (cnden .ne. 0.0) cn(LL) = cnnum /(xtrc*cnden)
C**********************************************************************C
200       continue
C**********************************************************************C
c
          call vclr ( tri,1,istop )

          do 230 LL=ISTART,ISTOP
             tri(LL)=cn(LL)
230       continue

C**********************************************************************C
        call vmov (tri, 1, itr(ITHWP1), 1, nsamp)

          call savew2(itr,ifmt_RecNum,l_RecNum, ln_RecNum,
     1                irec , TRACEHEADER)
          call savew2(itr,ifmt_TrcNum,l_TrcNum, ln_TrcNum,
     1                 IR  , TRACEHEADER)
          idi = idtot
          ili = iltot
          call savew2(itr,ifmt_DphInd,l_DphInd, ln_DphInd,
     1                idi  , TRACEHEADER)
          call savew2(itr,ifmt_LinInd,l_LinInd, ln_LinInd,
     1                ili  , TRACEHEADER)
          call savew2(itr,ifmt_StaCor,l_StaCor, ln_StaCor,
     1                 0   , TRACEHEADER)
          ivalmx = valmx
          ivalmy = valmy
          call savew2(itr,ifmt_CDPBCX,l_CDPBCX, ln_CDPBCX,
     1                ivalmx, TRACEHEADER)
          call savew2(itr,ifmt_CDPBCY,l_CDPBCY, ln_CDPBCY,
     1                ivalmy, TRACEHEADER)

        call wrtape(luout, itr, obytes)

100       continue
C**********************************************************************C
c
      first = .false.

      return
      end
C**********************************************************************C
C**********************************************************************C
      function rinter(y1,y2,dx)
C**********************************************************************C
c      interpolate between y1 and y2 for 0 < dx < 1
C**********************************************************************C
      rinter = (1.0 - dx)*y1 + dx*y2
      return
      end

c-----------------------------------------
c  online help section
c-----------------------------------------
       subroutine  help
#include <f77/iounit.h>

         write(LER,*)
     :'***************************************************************'
      write(LER,*)' '
      write(LER,*)
     :'rnmospec: compute semblances for residual NMO using gamma scan'
      write(LER,*)
     :'Execute program by typing rnmospec and the following arguments'
      write(LER,*)' '
      write(LER,*)
     :' -N [ntap]    (default stdin)             : Input data file name'
      write(LER,*)
     :' -O [otap]    (default stdout)            : Output data file'
      write(LER,*)' '
      write(LER,*)
     :' -tmin[start] (default = 0 ms) : start time of spectra (ms)'
      write(LER,*)
     :' -tmax [stop] (def = end trc)  :  stop time of spectra (ms)'
      write(LER,*)
     :' -rs [irs],   (default=1)      : beginning record number'
      write(LER,*)
     :' -re [ire],   (default=last)   : ending record number'
      write(LER,*)
     :' -ns[ns],     (default=1)      : starting trace number'
      write(LER,*)
     :' -ne[ne]      (default=last)   : ending trace number'
      write(LER,*)
     :' -nvel[nvel]  (default = 100)  : number of RNMO scans'
      write(LER,*)
     :' -igt[igt]    (default = 8)    : points in semblance wwindow'
      write(LER,*)
     :' -dmul [dmul] (default =1.0)   : distance multiplier'
      write(LER,*)
     :' -tmul [tmul] (default=1.0)    : sampling interval multiplier'
      write(LER,*)
     :' -rmin[rmin] (default = 950)   : min RNMO gamma value'
      write(LER,*)
     :' -rmax[rmax] (default = 1050)  : max RNMO gamma value'
        write(LER,*)
     :' -unit [unit] (default = 1000) : scale factor for gammas'
      write(LER,*)' '
      write(LER,*)
     :' -V          (default = false) : verbose output'
c     write(LER,*)
c    :' -A          (default = false) : absolute semblance contours'
      write(LER,*)
     :'              data dependent - for noisy data try omitting -A'
      write(LER,*)' '
      write(LER,*)
     :'USAGE: rnmospec -N[ntap] -O[otap] -ns[ns] -ne[ne] '
      write(LER,*)
     :'                -re[re] -rs[rs] -rmin[rmin] -rmax[rmax]'
      write(LER,*)
     :'                -nvel[nvel] -tmin[start] -tmax[stop] -igt[igt] '
      write(LER,*)
     :'                -unit[] -tmul[tmul] -dmul[dmul] -V'
      write(LER,*)' '
         write(LER,*)
     :'***************************************************************'

      return
      end

c-----
c     get command arguments
c
c     ntap  - C*100  input file name
c     otap  - C*100  output file name
c      ns   - I      start trace
c      ne   - I      stop end trace
c     irs   - I      start record
c     ire   - I      stop end record
c   start   - R      start time of analysis
c    stop   - R      stop time
c    dmul   - R      distance multiplier
c    tmul   - R      sample interval multiplier
c    nvel   - I      number velocities
c    vmin   - R      min RNMO
c    vmax   - R      max RNMO
c    abslut - L      use absolute semblance
c    verbos - L      verbose output or not
c-----
      subroutine cmdln (ntap,otap,start,stop,dmul,tmul,igt,xmin,xmax,
     &               ns,ne,irs,ire,nvel,vmin,vmax,abslut,verbos,
     &               iscl)
#include <f77/iounit.h>
#include <f77/lhdrsz.h>
      character  ntap*(*), otap*(*)
      integer    argis,ns,ne,irs,ire,nvel,igt,start,stop,iscl
      real       dmul,tmul,vmin,vmax
      logical    abslut, verbos

         call argstr('-N', ntap, ' ', ' ' )
         call argstr('-O', otap, ' ', ' ' )
         call argr4 ('-tmin', START, 0., 0.)
         call argr4 ('-tmax', STOP , 0., 0.)
         call argr4('-dmul',dmul,1.0,1.0)
         call argr4('-tmul',tmul,1.0,1.0)
         call argi4('-igt',IGT,8,8)
         call argi4 ('-rs',irs,1,1)
         call argi4 ('-re',ire,0,0)
         call argi4 ('-ns',ns,1,1)
         call argi4 ('-ne',ne,0,0)
         call argi4 ('-nvel',nvel,100,100)
         call argi4 ('-unit',iscl,1000,1000)
         call argr4('-rmin',vmin,950.,950.)
         call argr4('-rmax',vmax,1050.,1050.)

         if (vmin .eq. 0.0) then
            write(LERR,*)'Must enter minimum RNMO value -- FATAL'
            write(LERR,*)'Use -rmin[] on cmd line & rerun'
            write(LER ,*)'Must enter minimum RNMO value -- FATAL'
            write(LER ,*)'Use -rmin[] on cmd line & rerun'
            stop
         endif
         if (vmax .eq. 0.0) then
            write(LERR,*)'Must enter maximum RNMO value -- FATAL'
            write(LERR,*)'Use -rmax[] on cmd line & rerun'
            write(LER ,*)'Must enter maximum RNMO value -- FATAL'
            write(LER ,*)'Use -rmax[] on cmd line & rerun'
            stop
         endif
         verbos = ( argis ('-V') .gt. 0 )
         abslut = ( argis ('-A') .gt. 0 )
         if(nvel.gt.SZSPRD) then
            write(LERR,*)'Number velocities too high'
            write(LERR,*)'Must be less than ',SZSPRD
            stop
         endif

      return
      end
