c***********************************************************************
c                                                                      *
c                 copyright 2001, Amoco Production Company             *
c                             All Rights Reserved                      *
c                     an affiliate of BP America Inc.                  *
c                                                                      *
c***********************************************************************
c  routine:       rngtms                                               *
c  routine type:  function  real                                       *
c  purpose:                                                            *
c      implement the range trimmed mean algorithm, for stacking.       *
c      a statistical count is kept of the number of times the data     *
c      value from each input array element is included in the          *
c      stack.                                                          *
c                                                                      *
c  entry points:                                                       *
c      rngtms  real  (luprt,alphap,nfold,ntlive,b,livetr,istat,ipntr,  *
c                     tabul8)                                          *
c  arguments:                                                          *
c      luprt   integer       i           -                             *
c      alphap  real          i           -                             *
c      nfold   integer       i           -                             *
c      ntlive  integer       i           -                             *
c      b       real     ??iou*  (nfold) -                              *
c      livetr  logical  ??iou*  (nfold) -                              *
c      istat   integer  ??iou*  (nfold) -                              *
c      ipntr   integer  ??iou*  (nfold) -                              *
c      tabul8  logical  ??iou*           -                             *
c       +------------------------------------------------------+       *
c       |               development information                |       *
c       +------------------------------------------------------+       *
c  author:   bill done                          origin date: 86/09/19  *
c  language: fortran 77                  date last compiled: 86/09/19  *
c       +------------------------------------------------------+       *
c       |                 external environment                 |       *
c       +------------------------------------------------------+       *
c  routines called:                                                    *
c      srtqkp -                                                        *
c  intrinsic functions called:                                         *
c      float   real    -                                               *
c      amin1   real    -                                               *
c      mod     generic -                                               *
c  files:                                                              *
c      luprt  ( output sequential ) -                                  *
c  common:           none                                              *
c  stop codes:       none                                              *
c       +------------------------------------------------------+       *
c       |             other documentation details              |       *
c       +------------------------------------------------------+       *
c  error handling:                                                     *
c      if nfold or ntlive are less than 1, the function value is       *
c      set to zero.                                                    *
c                                                                      *
c  general description:                                                *
c      implement the range trimmed mean algorithm for finding the      *
c      representative value of nfold points in vector b.  parameter    *
c      alphap determines a percentage of the minimum range (minimum    *
c      from median to max or min). all members of b lying within that  *
c      partial range of the median of b are averaged to compute the    *
c      representation of the nfold points of b.  the remaining         *
c      points are ignored.                                             *
c                                                                      *
c      this results in a variable interval around the median and a     *
c      varying number of elements of b which are averaged to determine *
c      the trimmed mean.                                               *
c                                                                      *
c  revised by:  bill done                     revision date: 86/09/19  *
c      modified to eliminate need to sort dead trace values to rear    *
c      of b vector.                                                    *
c                                                                      *
c       +------------------------------------------------------+       *
c       |                 analysis information                 |       *
c       +------------------------------------------------------+       *
c  nonstandard features:   none detected                               *
c*******************   end of documentation package   ******************
c***********************************************************************
c
      real function rngtms (luprt , alphap, nfold , ntlive, b, livetr,
     *                      istat , ipntr , tabul8)
c
      logical livetr(nfold), oddntl, tabul8
      integer istat(nfold), ipntr(nfold)
      real b(nfold), median, medpq, medmq
      data iprint/0/
c
c     convert alpha from per cent to decimal.
c     print function name and value of alpha on first entry.
c
      if (iprint .eq. 0) then
         iprint = 1
         alpha = alphap/100.
         write (luprt,1200) nfold, alphap
 1200    format(/' RNGTMS:  nfold = ',i5,3x,'alpha = ',f7.2/)
      endif
c
c     set rngtms to zero if nfold < 1 or ntlive < 1
c
      if ((nfold .lt. 1) .or. (ntlive .lt. 1)) then
         rngtms = 0.0
         return
      endif
c
c     set rngtms to b(nfold) if nfold = 1.
c
      if (nfold .eq. 1) then
         rngtms = b(nfold)
         if (tabul8) istat(nfold) = istat(nfold) + 1
         return
      endif
c
c     sort the ntlive elements of b, which are at the front of b.
c
      call srtqkp(ntlive,b,nfold,ipntr)
c
c     find center index of array b (if ntlive is odd, nd2p1 is center;
c     if ntlive is even, nd2 and nd2p1 are the center 2 values)
c
      nd2 = ntlive/2
      nd2p1 = nd2 + 1
c
c     find the median value of the data in b
c
      oddntl = mod(ntlive,2) .eq. 1
      if (oddntl) then
         median = b(nd2p1)
       else
         median = 0.5*(b(nd2) + b(nd2p1))
      endif
c
c     compute the range of b and q, which determines the
c     interval around the median from which elements of b
c     are averaged
c
      range = amin1(b(ntlive)-median,median-b(1))
      q = alpha*range
c
c     calculate the median - q and median + q interval
c
      medmq = median - q
      medpq = median + q
c
c     sum the elements of b which have values in the
c     (median-q, median+q) interval.  at least one value
c     (median) is guaranteed to be in this interval, even
c     if alpha = 0.0.
c
c     if this time step is to be tabulated (tabul8 = .true.),
c     then the statistics vector is updated.  the pointer vector
c     is used to update the element in istat corresponding to
c     the data value being stacking in b.
c
      kcount = 0
      rngtms = 0.0
      do 200 k = 1, ntlive
         bk = b(k)
         if ((bk .ge. medmq) .and. (bk .le. medpq)) then
            kcount = kcount + 1
            rngtms = rngtms + bk
            if (tabul8) istat(ipntr(k)) = istat(ipntr(k)) + 1
         endif
  200 continue
c
c     normalize the stack by the number of elements in the stack.
c     in the situation where all data elements are outside the
c     median + and - q interval, set the stack equal to the median
c     and update the statistical count depending on whether the
c     the median is determined from an odd or even number of samples.
c
      if (kcount .gt. 0) then
         rngtms = rngtms/float(kcount)
       else
         rngtms = median
         if (tabul8) then
            if (oddntl) then
                istat(ipntr(nd2p1)) = istat(ipntr(nd2p1)) + 1
              else
                istat(ipntr(nd2)) = istat(ipntr(nd2)) + 1
                istat(ipntr(nd2p1)) = istat(ipntr(nd2p1)) + 1
            endif
         endif
      endif
      return
      end
