/***********************************************************************
 *                copyright 2001, Amoco Production Company             *
 *                            All Rights Reserved                      *
 *                    an affiliate of BP America Inc.                  *
 ***********************************************************************/
/*
  Modified by:  Cory Hoelting
  Original code by:  Chi-hsin Wu
  Date last modified:  July 29, 1996
  Purpose:  Calculate texture features/attributes that are used
            by disparity.
*/


#include <stdio.h>
#include <math.h>
#include <string.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <sys/uio.h>
#include <stdlib.h>
#include <memory.h>

/* USP stuff */
/* Used for command line processing. */
/* This code may be converted to work with the */
/* USP file format in the future. */
#include <iounit.h>
#include <lhdrsz.h>
#include <save_defs.h>
#include <cu_defs.h>
#include <sisio.h>

#define      Q4      4                                  /*no. of neighbors    */
#define      FR      5   /*no. of features/transformations                    */

/* The constant TEX_REG is the value that will need to be replaced */
/* with the command line parameter -attr_hw <ahw> (the half window */
/* of the attribute analysis window. */ 
#define      TEX_REG  3  /* half aperature of feature window */ 
#define      SQR(a) ((a)*(a))

/* constants used during scaling and histogram equalization */
#define MAX_GRAY_LEVEL 256
#define MGL_DOUBLE 256.0

void texstat2d_help();
void allocation();
void features();
void max_minVs();
void boundary_residual();
void h_dir_residual();
void v_dir_residual();
void readpixel();
void scale_and_equalize();
void write_record();

int    **interior; /* interior[i][j]=1 if inside image lattice, otherwise 0  */
int idelta[Q4]={1,0,-1,0};   /* four near neighbors that assoc. pixel block  */
int jdelta[Q4]={0,1,0,-1};   /* not overlap.  see Geman's PAMI'90  p.615     */

/* global variable for program name, used in error messages */
char prog_name[] = "texstat2d";

int main(argc,argv)
int argc;
char *argv[];
{
 int ifp, t1fp, t2fp, t3fp, t4fp, t5fp;
 int pipe1, pipe2, pipe3, pipe4, pipe5;
 int one;
 char input_file[256], t1_file[256], t2_file[256],
      t3_file[256], t4_file[256], t5_file[256];
 int t1_bool, t2_bool, t3_bool, t4_bool, t5_bool;
 char host[10], blank9[10];
 int ikp;
 int num_of_tfiles;  /* the number of attributes calculated, must be >= 1 */

 int    **obs;      /* the data (here noisless) */
 double **y;        /* the transformed data i.e., y=y(obs); */

 int ROW,COL;      /* size of row and column of pixel sites */

 int usp_lineheader[SZLNHD];
 float *usp_trace;
 int usp_trace_bytes;
 int **trace_headers;          /* Storage for USP trace headers */

 int hist_eq;                  /* flag which is set of one if scaling */
                               /* and histogram equalization is to be */
                               /* performed */

  /* Help? */
  if (C_ARGIS("-?", &argc, argv) ||
      C_ARGIS("-h", &argc, argv) ||
      C_ARGIS("-help", &argc, argv))
    {
     texstat2d_help();
     exit(0);
    }

  strcpy(host, "         "); 
  strcpy(blank9, "         "); 

  ikpchk_(host, strlen(host));
  if (strcmp(host, blank9))
     ikp = 1;
  else
     ikp = 0; 

  /* display usage part of help if there was */
  /* only one argument on the command line */ 
  if (argc == 1 && !ikp)
    {
     fprintf(stderr, 
          "Usage:  texstat2d [-N <input_file>] -t1 <tex_file1> -t2 <tex_file2>\n\
                  -t3 <tex_file3> -t4 <tex_file4> -t5 <tex_file5> [-histeq]\n\
                  [-h -? -help]\n");
     exit(0);
    }

  num_of_tfiles = 0; 

  /* process the command line arguments */
  C_ARGSTR("-N", input_file, "", "", &argc, argv);
  if (strlen(input_file) < 1)
     ifp = 0; 
  else
     lbopen(&ifp, input_file, "r");

  pipe1 = 1;
  pipe2 = 3;
  pipe3 = 4;
  pipe4 = 5;
  pipe5 = 6;
  one = 1;

  C_ARGSTR("-t1", t1_file, "", "", &argc, argv);
  if (strlen(t1_file) < 1)
    {
     if (ikp)
       {
        /* begin debug */
        /*
        fprintf(stderr, "before the pipcnt\n");
        */
        /* end debug */

        if (pipcnt_(&pipe1,&one) > 0)
          {
           t1_bool = 1;
           sisfdfit_(&t1fp,&pipe1);
          }
        else
           t1_bool = 0;
       }
     else
        t1_bool = 0;
    }
  else
    {
     lbopen(&t1fp, t1_file, "w");
     t1_bool = 1;
    }

  C_ARGSTR("-t2", t2_file, "", "", &argc, argv);
  if (strlen(t2_file) < 1)
    {
     if (ikp)
       {
        if (pipcnt_(&pipe2,&one) > 0)
          {
           t2_bool = 1;
           sisfdfit_(&t2fp,&pipe2);
          }
        else
           t2_bool = 0;
       }
     else
        t2_bool = 0;
    }
  else
    {
     lbopen(&t2fp, t2_file, "w");
     t2_bool = 1;
    }

  C_ARGSTR("-t3", t3_file, "", "", &argc, argv);
  if (strlen(t3_file) < 1)
    {
     if (ikp)
       {
        if (pipcnt_(&pipe3,&one) > 0)
          {
           t3_bool = 1;
           sisfdfit_(&t3fp,&pipe3);
          }
        else
           t3_bool = 0;
       }
     else
        t3_bool = 0;
    }
  else
    {
     lbopen(&t3fp, t3_file, "w");
     t3_bool = 1;
    }

  C_ARGSTR("-t4", t4_file, "", "", &argc, argv);
  if (strlen(t4_file) < 1)
    {
     if (ikp)
       {
        if (pipcnt_(&pipe4,&one) > 0)
          {
           t4_bool = 1;
           sisfdfit_(&t4fp,&pipe4);
          }
        else
           t4_bool = 0;
       }
     else
        t4_bool = 0;
    }
  else
    {
     lbopen(&t4fp, t4_file, "w");
     t4_bool = 1;
    }

  C_ARGSTR("-t5", t5_file, "", "", &argc, argv);
  if (strlen(t5_file) < 1)
    {
     if (ikp)
       {
        if (pipcnt_(&pipe5,&one) > 0)
          {
           t5_bool = 1;
           sisfdfit_(&t5fp,&pipe5);
          }
        else
           t5_bool = 0;
       }
     else
        t5_bool = 0;
    }
  else
    {
     lbopen(&t5fp, t5_file, "w");
     t5_bool = 1;
    }

  hist_eq = C_ARGIS("-histeq", &argc, argv);

  num_of_tfiles = t1_bool + t2_bool + t3_bool + t4_bool + t5_bool;

  if (num_of_tfiles == 0)
    {
     fprintf(stderr, "%s: error: at least one output file must be specified\n",
             prog_name);
     exit(1);
    } 

  /* begin debug */
  /*
  fprintf(stderr, "before RTAPE\n");
  fprintf(stderr, "t1_bool = %d\n", t1_bool);
  fprintf(stderr, "t2_bool = %d\n", t2_bool);
  fprintf(stderr, "t3_bool = %d\n", t3_bool);
  fprintf(stderr, "t4_bool = %d\n", t4_bool);
  fprintf(stderr, "t5_bool = %d\n", t5_bool);
  */
  /* end debug */

  /* Read in the USP lineheader */
  C_RTAPE(ifp, usp_lineheader, &usp_trace_bytes);

  /* Grab values from the USP lineheader */
  C_SAVER(usp_lineheader, "NumSmp", &COL, LINEHEADER);
  C_SAVER(usp_lineheader, "NumTrc", &ROW, LINEHEADER);

  /* Update the historical line header */
  /* savhlh_(usp_lineheader, &usp_trace_bytes, &usp_trace_bytes); */

  if (t1_bool)
     C_WRTAPE(t1fp, usp_lineheader, usp_trace_bytes);
  if (t2_bool)
     C_WRTAPE(t2fp, usp_lineheader, usp_trace_bytes);
  if (t3_bool)
     C_WRTAPE(t3fp, usp_lineheader, usp_trace_bytes);
  if (t4_bool)
     C_WRTAPE(t4fp, usp_lineheader, usp_trace_bytes);
  if (t5_bool)
     C_WRTAPE(t5fp, usp_lineheader, usp_trace_bytes);

  /*
  fprintf(stderr, "before usp_trace_btyes = ...\n");
  */

  usp_trace_bytes = SZDTHD + (COL * SZSAMP);
  allocation(ROW, COL, &obs, &y, &trace_headers);
  readpixel(ifp, ROW, COL, obs, trace_headers, usp_trace_bytes, hist_eq);
  features(ROW, COL, t1_bool, t1fp, t2_bool, t2fp, t3_bool, t3fp,
           t4_bool, t4fp, t5_bool, t5fp, obs, y, trace_headers, usp_trace_bytes);

  /* close files */
  lbclos(ifp);
  if (t1_bool)
     lbclos(t1fp);
  if (t2_bool)
     lbclos(t2fp);
  if (t3_bool)
     lbclos(t3fp);
  if (t4_bool)
     lbclos(t4fp);
  if (t5_bool)
     lbclos(t5fp);

  /* Normal completion */
  fprintf(stderr, "%s:  normal completion\n", prog_name);

return(0);
}/*main*/
 

/* allocate memory */
void allocation(ROW, COL, obs, y, trace_headers)
int ROW;
int COL;
int ***obs;
double ***y;
int ***trace_headers;
{
  int i,j,nn,fr;
  double ***pccc, **pcc, *pc;
  int    ***ipccc, **ipcc, *ipc;
                                                
  /* If TEX_REG is changed from a constant to a variable whose value */
  /* is obtained from the command line, then the proper amount of */
  /* memory will be allocated so an out of bounds error will not occur. */
  interior=(int **)malloc((ROW+2*TEX_REG)*sizeof(int *));   /*allocate interior*/
  interior += TEX_REG;
  ipc=(int *)malloc((ROW+2*TEX_REG)*(COL+2*TEX_REG)*sizeof(int));
  ipc += TEX_REG*(COL+2*TEX_REG)+TEX_REG;
  for (i=-TEX_REG; i<ROW+TEX_REG; i++)
    interior[i]=ipc+i*(COL+2*TEX_REG);
  for (i=-TEX_REG;i<ROW+TEX_REG;i++)  /*initialize interior[][]*/
    for (j=-TEX_REG;j<COL+TEX_REG;j++) {
      if ( (i >0) && (i < ROW-1) && (j >0) && (j < COL-1) )
        interior[i][j]=4;
      else if (i<0 || j<0 || i>=ROW || j>=COL)
        interior[i][j]=0;
      else if ((i==0 && (j==0 || j==COL-1)) || (i==ROW-1 && (j==0 || j==COL-1)))
        interior[i][j]=2; 
      else 
        interior[i][j]=3;
    }/*for j*/

  /*2D array of data*/
  (*obs)=(int **)malloc(ROW*sizeof(int *));
  ipc=(int *)malloc(ROW*COL*sizeof(int));
  for (i=0; i<ROW; i++)
    (*obs)[i]=ipc+i*COL;

  /*2D array of transformed data */
  (*y)=(double **)malloc(ROW*sizeof(double *));
  pc=(double *)malloc(ROW*COL*sizeof(double));
    for (i=0; i<ROW; i++)
      (*y)[i]=pc+i*COL;

  /* Storage for the USP trace headers */
  *trace_headers = (int **) malloc(sizeof(int *) * ROW);
  for(i = 0; i < ROW; i++)
     (*trace_headers)[i] = (int *) malloc(SZDTHD);

}/*allocation*/


/*find the features/transformation of data and write them to files */
void features(ROW, COL, t1_bool, t1fp, t2_bool, t2fp, t3_bool, t3fp,
              t4_bool, t4fp, t5_bool, t5fp, obs, y, trace_headers, 
              usp_trace_bytes)
int ROW, COL, t1_bool, t2_bool, t3_bool, t4_bool, t5_bool;
FILE *t1fp, *t2fp, *t3fp, *t4fp, *t5fp;
int **obs;
double **y;
int **trace_headers;
int usp_trace_bytes;
{
  int i,j;

  if (t1_bool)
    {
     /*feature #0: data itself*/
     /* no-op */
     fprintf(stderr, "%s: processing feature/attribute 1\n", prog_name);
     write_record(ROW, COL, t1fp, obs, NULL, trace_headers, usp_trace_bytes);
     /*
     for(i=0; i<ROW; i++)
       for(j=0; j<COL; j++)
          fprintf(t1fp, "%f ", (double) obs[i][j]);
     */
    }

  if (t2_bool)
    {   
     /*feature #1: intensity range*/
     fprintf(stderr, "%s: processing feature/attribute 2\n", prog_name);
     max_minVs(ROW, COL, obs, y);
     write_record(ROW, COL, t2fp, NULL, y, trace_headers, usp_trace_bytes);
     /*
     for(i=0; i<ROW; i++)
       for(j=0; j<COL; j++)
          fprintf(t2fp, "%f ", y[i][j]);
     */
    }
   
  if (t3_bool)
    {
     /*feature #2: boundary residual*/
     fprintf(stderr, "%s: processing feature/attribute 3\n", prog_name);
     boundary_residual(ROW, COL, obs, y);
     write_record(ROW, COL, t3fp, NULL, y, trace_headers, usp_trace_bytes);
     /*
     for(i=0; i<ROW; i++)
       for(j=0; j<COL; j++)
          fprintf(t3fp, "%f ", y[i][j]);
     */
    }

  if (t4_bool)
    {
     /*feature #3: horiaontal directional residual*/
     fprintf(stderr, "%s: processing feature/attribute 4\n", prog_name);
     h_dir_residual(ROW, COL, obs, y);
     write_record(ROW, COL, t4fp, NULL, y, trace_headers, usp_trace_bytes);
     /*
     for(i=0; i<ROW; i++)
       for(j=0; j<COL; j++)
          fprintf(t4fp, "%f ", y[i][j]);
     */
    }

  if (t5_bool)
    {
     /*feature #4: vertical directional residual*/
     fprintf(stderr, "%s: processing feature/attribute 5\n", prog_name);
     v_dir_residual(ROW, COL, obs, y);
     write_record(ROW, COL, t5fp, NULL, y, trace_headers, usp_trace_bytes);
     /*
     for(i=0; i<ROW; i++)
       for(j=0; j<COL; j++)
          fprintf(t5fp, "%f ", y[i][j]);
     */
    }


} /* features */


/*find the max-min value of pixel in the (2*TEX_REG+1)x(2*TEX_REG+1) block, Vs,
 *centered at (i,j) for all ordered pairs in the ROWxCOL lattice.  */
void max_minVs(ROW, COL, obs, y)
int ROW, COL; 
int **obs;
double **y;
{
  int i, j, m,n;
  int maxval,minval;

  for(i = 0; i < ROW; i++)
     for(j = 0; j < COL; j++)
       {
        maxval=minval=obs[i][j];
        for (m=-TEX_REG+i; m<=TEX_REG+i; m++) 
          for (n=-TEX_REG+j; n<=TEX_REG+j; n++) 
            if (interior[m][n])
              {
               if (obs[m][n]>maxval) maxval=obs[m][n];
               if (obs[m][n]<minval) minval=obs[m][n];
              }
        y[i][j] = (maxval-minval);
       }

}/*max_minVs*/


/* find the residual by comparing obs[i][j] to the boundary of Vs */
void boundary_residual(ROW, COL, obs, y)
int ROW, COL;
int **obs;
double **y;
{
  int numbd; /*total number of boundary pixels of Vs*/
  double sum;
  int i, j, m,n;

  for(i = 0; i < ROW; i++)
     for(j = 0; j < COL; j++) 
       {
        numbd=0;
        sum=0.0;
        for (m=-TEX_REG+i; m<=TEX_REG+i; m += 2*TEX_REG) 
          for (n=-TEX_REG+j; n<=TEX_REG+j; n++) 
            if (interior[m][n]) {
              sum += obs[m][n];
              numbd++;
            }
        for (n=-TEX_REG+j; n<=TEX_REG+j; n += 2*TEX_REG) 
          for (m=-TEX_REG+i+1; m<=TEX_REG+i-1; m++)
            if (interior[m][n]) {
              sum += obs[m][n];
              numbd++;
            }
        y[i][j] = (fabs(obs[i][j]-sum/numbd));        
       }

}/*boundary_residual*/


/* find the horizontal directional residual */
void h_dir_residual(ROW, COL, obs, y)
int ROW, COL;
int **obs;
double **y;
{
  int i, j, m,n,dir,num;
  double sum;
  
  for(i = 0; i < ROW; i++)
     for(j = 0; j < COL; j++)
       {
        num=0;
        sum=0.0;
        for (dir=0; dir<Q4; dir += 2) {
          m=i+idelta[dir]; n=j+jdelta[dir];
          if (interior[m][n]) {
            sum += obs[m][n];
            num++;
          }/*if*/
        }/*for dir*/
        y[i][j] = (fabs(obs[i][j]-(sum/num)));  
       }

}/*h_dir_residual*/


/* find the horizontal directional residual */
void v_dir_residual(ROW, COL, obs, y)
int ROW, COL;
int **obs;
double **y;
{
  int i, j, m,n,dir,num;
  double sum;
  
  for(i = 0; i < ROW; i++)
     for(j = 0; j < COL; j++)
       {
        num=0;
        sum=0.0;
        for (dir=1; dir<Q4; dir += 2) {
          m=i+idelta[dir]; n=j+jdelta[dir];
          if (interior[m][n]) {
            sum += obs[m][n];
            num++;
          }
        }
        y[i][j] = (fabs(obs[i][j]-(sum/num)));  
       }

}/*v_dir_residual*/


/*read image from the file fp*/
void readpixel(fp, ROW, COL, img, trace_headers, usp_trace_bytes, hist_eq)
int fp; /* input file */
int ROW;
int COL;
int **img;
int **trace_headers;
int usp_trace_bytes;
int hist_eq;
{
  float *usp_trace;
  int i, j, k;

  usp_trace = (float *) malloc(usp_trace_bytes);
  for(i = 0; i < ROW; i++)
    {
     /* Read the next trace */
     C_RTAPE(fp, usp_trace, &usp_trace_bytes);
     memcpy(trace_headers[i], usp_trace, ITRWRD * sizeof(int));
     /* memcpy(img[i], &(usp_trace[ITRWRD]), COL * sizeof(int)); */
     for(j = 0, k = ITRWRD; j < COL; j++, k++)
        img[i][j] = (int) usp_trace[k];
    } 
  free(usp_trace); 

  /* if the hist_eq flag is set, scale and equalize the histogram */
  if (hist_eq)
     scale_and_equalize(ROW, COL, img);
}


/* this function scales the data values between 0...255 */
void scale_and_equalize(ROW, COL, img)
 int ROW, COL;
 int **img;
 {
  double pdf[MAX_GRAY_LEVEL], cdf[MAX_GRAY_LEVEL];
  double num, den, norm;              /* temp variables */
  int i, j, max_amp, min_amp;

  /* Find the minimum and maximum values. */
  max_amp = -999999;
  min_amp = 999999;
  for(i = 0; i < ROW; i++)
     for(j = 0; j < COL; j++)
       {
        if (img[i][j] > max_amp)
           max_amp = img[i][j];
        if (img[i][j] < min_amp)
           min_amp = img[i][j];
       }

  /* Scale the data values between 0...255 */
  den = (double) (max_amp - min_amp); 
  for(i = 0; i < ROW; i++)
     for(j = 0; j < COL; j++)
       {
        num = (double) (img[i][j] - min_amp);
        img[i][j] = (int) ( (MGL_DOUBLE - 1.0) * (num / den) ); 
       }

  /* Calculate the PDF and CDF of the data set. */
  for(i = 0; i < MAX_GRAY_LEVEL; i++)
    {
     pdf[i] = 0.0;
     cdf[i] = 0.0;
    }
 
  norm = 1.0 / ( ((double) ROW) * ((double) COL) );
 
  for(i = 0; i < ROW; i++)
     for(j = 0; j < COL; j++)
        pdf[ img[i][j] ] += 1.0;
 
  for(i = 0; i < MAX_GRAY_LEVEL; i++)
     pdf[i] *= norm;
 
  cdf[0] = pdf[0];
  for(i = 1; i < MAX_GRAY_LEVEL; i++)
     cdf[i] = pdf[i] + cdf[i-1];

  /* Equalize the histogram */
  for(i = 0; i < ROW; i++)
     for(j = 0; j < COL; j++)
        img[i][j] = (int)((MGL_DOUBLE - 1.0) * cdf[ img[i][j] ]);
 }


void write_record(ROW, COL, tfp, obs, y, trace_headers, usp_trace_bytes)
int ROW, COL, tfp;
int **obs;
double **y;
int **trace_headers;
int usp_trace_bytes;
{
 float *usp_trace;
 int i, j, k;

 usp_trace = (float *) malloc(usp_trace_bytes);
 if (obs)
   {
    for(i = 0; i < ROW; i++)
      {
       memcpy(usp_trace, trace_headers[i], ITRWRD * sizeof(float));
       /* memcpy(&(usp_trace[ITRWRD]), obs[i], COL * sizeof(float)); */
       for(j = 0, k = ITRWRD; j < COL; j++, k++)
          usp_trace[k] = (float) obs[i][j];
       C_WRTAPE(tfp, usp_trace, usp_trace_bytes);
      }
   }
 if (y) 
   {
    for(i = 0; i < ROW; i++)
      {
       memcpy(usp_trace, trace_headers[i], ITRWRD * sizeof(float));
       for(j = 0, k = ITRWRD; j < COL; j++, k++)
          usp_trace[k] = (float) y[i][j]; 
       C_WRTAPE(tfp, usp_trace, usp_trace_bytes);
      }
   }
 free(usp_trace);
}


void texstat2d_help()
 {
  fprintf(stderr, "\
  This program computes attributes used to distinguish textures in\n\
  a 2D data set.  An example of a 2D data set is one frequency in the\n\
  spectral decomposition of a horizon.  The data values can be scaled\n\
  between 0 and 255 and the histogram of the data values can be equalized.\n\
  The output of this program is an N vector at each data site in the input\n\
  file.  For a data site the N vector contains the attributes calculated\n\
  from the data values in the attribute analysis window around that data\n\
  site.  At the present time, any combination of the five attributes used by\n\
  Wu & Doerschuk can be computed by this program.  The combination of\n\
  attributes computed by this program is determined by the command line\n\
  options.  Each attribute which is computed is written into a different\n\
  file.\n\
  REFERENCE:  C. Wu and P. Doerschuk, Texture-based segmentation using\n\
              Markov random field models and approximate Bayesian\n\
              estimators based on trees, Jrnl. of Math. Imaging and Vision,\n\
              vol. 5, no. 4, Decemeber, 1995.\n\
  \n\
  Usage:  texstat2d [-N <input_file>] -t1 <tex_file1> -t2 <tex_file2>\n\
                    -t3 <tex_file3> -t4 <tex_file4> -t5 <tex_file5>\n\
                    [-histeq]\n\
                    -? -h -help\n\
  \n\
          -N <input_file> - the input file containing the 2D data set.\n\
                            The data set is often referred to as the data\n\
                            lattice.\n\
                            (Default: stdin)\n\
  \n\
          -t1 <tex_file1> - the output file for the first attribute of all\n\
                            the sites in the data lattice.  The first\n\
                            attribute is the data value at the data site.\n\
  \n\
          -t2 <tex_file2> - the output file for the second attribute of all\n\
                            the sites in the data lattice.  The second\n\
                            attribute is the intensity range, i.e. the\n\
                            maxval - minval in the attribute analysis window\n\
                            of the data site.\n\
  \n\
          -t3 <tex_file3> - the output file for the third attribute of all\n\
                            the sites in the data lattice.  The third\n\
                            attribute is the boundary residual, i.e. the\n\
                            data value at the data site minus the average\n\
                            of the data values on the boundary of the\n\
                            the data site's attribute analysis window.\n\
  \n\
          -t4 <tex_file4> - the output file for the fourth attribute of all\n\
                            the sites in the data lattice.  The fourth\n\
                            attribute is the horizontal directional\n\
                            residual, i.e. a gradient approximation in the\n\
                            horizontal direction.\n\
  \n\
          -t5 <tex_file5> - the output file for the fifth attribute of all\n\
                            the sites in the data lattice.  The fifth\n\
                            attribute is the vertical directional\n\
                            residual, i.e. a gradient approximation in the\n\
                            vertical direction.\n\
  \n\
  NOTE: If only two attributes are to be computed, then only the\n\
  corresponding attribute files are specified on the command line.\n\
  For example, if the flags -t1 <t1_file> and -t3 <t3_file> are specified\n\
  on the command line, only the first and third attributes are computed.\n\
  At least one attribute file must be specified on the command line.\n\
  \n\
          -histeq - when this flag is set the data values are scaled between\n\
                    0 and 255 and their gray level histogram is equalized\n\
                    before the attributes are calculated.\n\
  \n\
          -?, -h, -help - Help.\n\
");
}
