c...|....1....|....2....|....3....|....4....|....5....|....6....|....7..
C***********************************************************************
C  Copyright 2002, Allied Geophysics, Inc.   All Rights Reserved       *
C***********************************************************************
C Portions of this code and/or subroutines used by this code are       *
C protected by the following copyright(s):                             *
C***********************************************************************
C                 copyright 2001, Amoco Production Company             *
C                             All Rights Reserved                      *
C                     an affiliate of BP America Inc.                  *
C***********************************************************************
C Use of this program and source code is subject to the terms and      *
C conditions of the FreeUSP License agreement.                         *
C***********************************************************************
c -----------------  Main Routine -----------------------
c
c uspmath - perform unary math operations between multiple usp datasets
c
c  Program Description:
c
c    uspmath reads multiple usp datasets with or without embedded
c      mask vaules and performs unary math operations on the live
c      samples, writing the result to a single output.
c
c  Program History:
c    Jan 21, 2002 - Now use subroutine close_files() on errors
c    Jan 16, 2002 - Minor documentation changes
c    Jan 15, 2002 - Changed default mask behavior to be off
c    Jan  7, 2002 - Implemented -F as a way of getting input names
c                   Added hooks for ikp
c    Dec 31, 2001 - minor cleanup of code and man page
c    Oct 30, 2001 - adopted emask as name for embedded mask
c    Oct 29, 2001 - original version
c
c
c get machine dependent parameters 
#include <f77/iounit.h>
#include <f77/lhdrsz.h>
#include <f77/sisdef.h>
#include <save_defs.h> 

c Set the maximum number of inputs allowed
      integer    MAXFILES
      parameter (MAXFILES=1000)

c Declare standard USP variables 
      integer     ilhd(3*SZLNHD)

      integer     luin(MAXFILES), luout
      integer     obytes, lbytes, lbyout, nbytes
      integer     n1,n2,n3
      integer     argis

      character*255 ntap(MAXFILES)
      character     otap*255, name*7

      logical     verbose

c Program Specific - dynamic memory variables
      integer TrcSize
      integer TotalMem
      integer ier1,ier2,abort

      integer Trc_in(2), Trc_out(2)

      pointer (ptr_Trc_in,  Trc_in)
      pointer (ptr_Trc_out, Trc_out)

c Program Specific - static memory variables
      integer n1_1,n2_1,n3_1
      integer nfiles, ifile
      integer i2,i3, i,adjust
      real    emask, scalar(MAXFILES), bias(MAXFILES)
      real    minval,maxval, oldval,newval, epsilon

      character mode*6
      logical lmask, likp

      integer len,nblen, pipes(MAXFILES),pipcnt

c Initialize variables
      data abort/0/
      data name/"USPMATH"/

c Give command line help if requested
      if ( argis('-?')    .gt. 0 .or. 
     :     argis('-h')    .gt. 0 .or.
     :     argis('-help') .gt. 0 ) then
        call help(name)
        stop
      endif

c Open printout file
#include <f77/open.h>

c Get command line input parameters
      call cmdln (ntap, nfiles, otap, mode, scalar, bias, epsilon,
     :            minval,maxval, oldval,newval, emask,lmask, likp,
     :            name, verbose)

c Open input files
      if (.not.likp) then

        do ifile = 1,nfiles
          call getln(luin(ifile),ntap(ifile),'r',0)
        enddo

        if (verbose) then
          write(LERR,*)' '
          write(LERR,*)name,': nfiles= ',nfiles
          do ifile = 1,nfiles
            len=nblen(ntap(ifile))
            write(LERR,*)name,
     :        ': File ',ntap(ifile)(1:len),' is on unit ',luin(ifile)
          enddo
        endif

      else

c       Here we handle multiple input pipes that can come with ikp.
c
c   Beware !!! This stuff isn't really working. Fortunately, it's only
c              used when run from XIKP, and XIKP hasn't been directed
c              to set it up yet.   -mjo
c
c       pipcnt() finds the number of units listed in pipes() that are
c       open. nfiles+1 units are tested because /dev/null is on an
c       unknown unit that might be in the range of nfiles. pipcnt()
c       specifically squeezes the unit of /dev/null from the pipes()
c       array, but no others. Initially the pipes() array skips over
c       units that we know not to count as potential inputs. These
c       are 1, 2, and LERR.
c
c       sisfdfit() assigns input units and insures the underlying uspio
c       structures have the proper file descriptors. This allows use of
c       rtape() and wrtape() with files that weren't opened by getln()
c       or lbopen().

        adjust = 1
        pipes(1) = 0
        write(LERR,*)'nfiles = ',nfiles
        do i = 2,nfiles+1
          if (i.eq.LERR) adjust = adjust+1
          pipes(i) = i+adjust
        enddo

        nfiles = pipcnt (pipes,nfiles+1)

        write(LERR,*)'Number of pipes found= ',nfiles
        write(LERR,*)'socket numbers:'
        write(LERR,*)(pipes(i),i=1,nfiles)

        do i = 1,nfiles
          if (pipes(i) .eq. 0) then
            luin(i) = 0
          else
            call sisfdfit (luin(i), pipes(i))
          endif
          write(LERR,*)name,': fortran unit number= ',luin(i),
     :                      ', socket= ',pipes(i)
        enddo

      endif

c Open the output file
      call getln(luout,otap,'w',1)

c Read input line header for the first file and save certain parameters
      call rtape(luin(1),ilhd,lbytes)
      if(lbytes.eq.0)then
        len=nblen(ntap(1))
        write(LER,*)name,': No line header on input ',ntap(1)(1:len)
        write(LER,*)'FATAL'
        call close_files(luin,nfiles,luout)
        stop
      endif

c Print HLH to printout file
      call hlhprt (ilhd, lbytes, name, 4, LERR)

c Save out hlh and line header
      call savhlh (ilhd, lbytes, lbyout)
      call wrtape (luout, ilhd, lbyout)

c Get sizes for the first input
      call saver(ilhd, 'NumSmp', n1_1, LINHED)
      call saver(ilhd, 'NumTrc', n2_1, LINHED)
      call saver(ilhd, 'NumRec', n3_1, LINHED)

      if (verbose) then
        write(LER,*)' '
        write(LER,*)name,': Sizes for file ',ntap(1)(1:nblen(ntap(1))),
     :              ' are ',n1_1,n2_1,n3_1
      endif

c Go through the other inputs and make sure they're all conformable
      do ifile = 2,nfiles
        call rtape(luin(ifile),ilhd,lbytes)
        if(lbytes.eq.0)then
          len=nblen(ntap(ifile))
          write(LER,*)name,
     :      ': no line header on input ',ntap(ifile)(1:len)
          write(LER,*)'FATAL'
          call close_files(luin,nfiles,luout)
          stop
        endif

c Print HLH to printout file
        call hlhprt (ilhd, lbytes, name, 4, LERR)

c Get sizes and check for conformance
        call saver(ilhd, 'NumSmp', n1, LINHED)
        call saver(ilhd, 'NumTrc', n2, LINHED)
        call saver(ilhd, 'NumRec', n3, LINHED)

        if (verbose) then
          write(LER,*)name,': Sizes for file ',
     :                ntap(ifile)(1:nblen(ntap(ifile))),
     :                ' are ',n1,n2,n3
        endif

        if (n1.ne.n1_1 .or. n2.ne.n2_1 .or. n3.ne.n3_1) then
          write(LER,*)name,
     :      ': dataset in not conformable with first input',ntap(ifile)
          write(LER,*)'FATAL'
          call close_files(luin,nfiles,luout)
          stop
        endif

      enddo

c Just in case there's only one input
      n1 = n1_1
      n2 = n2_1
      n3 = n3_1

c Okay, now we can allocate traces
      TrcSize = SZTRHD + n1*SZSMPD

      TotalMem = 2*TrcSize

c     number bytes on input/output
      obytes = TrcSize

c Dynamic memory allocation:  
      call galloc(ptr_Trc_in,  TrcSize, ier1, abort)
      call galloc(ptr_Trc_out, TrcSize, ier2, abort)
    
      if (ier1.ne.0 .or. ier2.ne.0) then
        write(LERR,*)' '
        write(LERR,*)name,
     :    'Unable to allocate workspace: ',TotalMem,' bytes'
        write(LERR,*)'FATAL'
        write(LERR,*)' '
        write(LER,*)' '
        write(LER,*)name,
     :    'Unable to allocate workspace: ',TotalMem,' bytes'
        write(LER,*)'FATAL'
        write(LER,*)' '
        call close_files(luin,nfiles,luout)
        stop
      else
        write(LERR,*)' '
        write(LERR,*)name,'Allocated ',TotalMem,' bytes of workspace'
        write(LERR,*)' '
        if (verbose) then
          write(LER,*)' '
          write(LER,*)name,': Allocated ',TotalMem,' bytes of workspace'
        endif
      endif

c Verbose output of all pertinent information before processing begins
      call verbal
     :        (ntap, nfiles, otap, mode, scalar,bias,epsilon,
     :         n1,n2,n3, minval,maxval, oldval,newval,
     :         emask,lmask, verbose)

c BEGIN PROCESSING 
      if (verbose) then
        write(LER,*)' '
        write(LER,*)name,': Begin trace processing'
      endif

c     Loop over input records
      do i3 = 1,n3

c       Loop over input traces
        do i2 = 1,n2

c         Loop over input files
          do ifile = 1,nfiles

c           read a trace
            nbytes = 0
            if (ifile.eq.1) then
              call rtape(luin(ifile), Trc_out, nbytes)
            else
              call rtape(luin(ifile), Trc_in, nbytes)
            endif

c           trap read errors
            if(nbytes.eq.0)then
              if (verbose) then
                write(LER,*)' '
                write(LER,*)name,': Read error on input file ',ifile,
     :            ' at record = ',i3,' trace= ',i2
              endif
              write(LERR,*)name,': Read error on input ',ifile,
     :          ' at record = ',i3,' trace= ',i2
              call close_files(luin,nfiles,luout)
              stop
            endif

c           process the trace
            if (mode(1:3).eq.'sum') then
              call sum_fxn
     :               (Trc_in(ITHWP1),Trc_out(ITHWP1),n1,
     :                scalar(ifile),bias(ifile),lmask,emask,ifile)

            elseif (mode(1:4).eq.'mult') then
              call mult_fxn
     :               (Trc_in(ITHWP1),Trc_out(ITHWP1),n1,
     :                scalar(ifile),bias(ifile),lmask,emask,ifile)

            elseif (mode(1:3).eq.'div') then
              call div_fxn
     :               (Trc_in(ITHWP1),Trc_out(ITHWP1),n1,
     :                scalar(ifile),bias(ifile),lmask,emask,ifile)

            elseif (mode(1:3).eq.'min') then
              call min_fxn
     :               (Trc_in(ITHWP1),Trc_out(ITHWP1),n1,
     :                scalar(ifile),bias(ifile),lmask,emask,ifile)

            elseif (mode(1:3).eq.'max') then
              call max_fxn
     :               (Trc_in(ITHWP1),Trc_out(ITHWP1),n1,
     :                scalar(ifile),bias(ifile),lmask,emask,ifile)

            elseif (mode(1:5).eq.'union') then
              call union_fxn
     :               (Trc_in(ITHWP1),Trc_out(ITHWP1),n1,
     :                scalar(ifile),bias(ifile),lmask,emask,ifile)

            elseif (mode(1:4).eq.'clip') then
              call clip_fxn
     :               (Trc_out(ITHWP1),n1,minval,maxval,
     :                scalar(ifile),bias(ifile),lmask,emask)

            elseif (mode(1:6).eq.'change') then
              call change_fxn
     :               (Trc_out(ITHWP1),n1,oldval,newval,
     :                scalar(ifile),bias(ifile),lmask,emask)

            elseif (mode(1:5).eq.'scale') then
              call scale_trc
     :               (Trc_out(ITHWP1),n1,
     :                scalar(ifile),bias(ifile),lmask,emask)
            endif

c         end of loop over input files
          enddo

c         write the trace
          call wrtape (luout, Trc_out, obytes)

c       end of loop over axis 2 (traces)
        enddo

c     end of loop over axis 3 (records)
      enddo


c Close data files 
      call close_files(luin,nfiles,luout)
      write(LERR,*)name,': Normal Termination'
      write(LER,*)name,': Normal Termination'
      stop

      end


c...|....1....|....2....|....3....|....4....|....5....|....6....|....7..
c Close all the inputs and the output
      subroutine close_files(luin,nfiles,luout)
      implicit none
      integer  nfiles,luin(nfiles),luout,ifile
      do ifile = 1,nfiles
        call lbclos(luin(ifile))
      enddo
      call lbclos (luout)
      return
      end


c...|....1....|....2....|....3....|....4....|....5....|....6....|....7..
c Provide terse online help [detailed help goes in man page]
      subroutine help(name)

      character name*(*)

#include <f77/iounit.h>

      write(LER,*)' '
      write(LER,*)'===================================================='
      write(LER,*)' '
      write(LER,*)' Command Line Arguments for ',name
      write(LER,*)'    Unary math operations between many USP datasets'
      write(LER,*)' '
      write(LER,*)'Input......................................... (def)'
      write(LER,*)' '
      write(LER,*)'-N[]    -- Input datasets                     (none)'
      write(LER,*)'             Repeat as many times as needed for'
      write(LER,*)'             multiple inputs. Use stdin: if first'
      write(LER,*)'             of many is a pipe. May be omitted if'
      write(LER,*)'             there is only one input on a pipe.'
      write(LER,*)'... or'
      write(LER,*)'-F[ftap] -- file of input file names, one per line'
      write(LER,*)'            (see man page)'
      write(LER,*)' '
      write(LER,*)'-O[]    -- Output dataset                  (stdout:)'
      write(LER,*)' '
      write(LER,*)'-mode[] -- Mode of operation                   (sum)'
      write(LER,*)'           (each operation follows scaling)'
      write(LER,*)' '
      write(LER,*)'           For multiple inputs'
      write(LER,*)'              sum    - sum inputs'
      write(LER,*)'              mult   - multiply inputs'
      write(LER,*)'              div    - divide inputs'
      write(LER,*)'              min    - take the minimum of inputs'
      write(LER,*)'              max    - take the maximum of inputs'
      write(LER,*)'              union  - create union of live samples'
      write(LER,*)' '
      write(LER,*)'           For single input'
      write(LER,*)'              clip   - clip live samples to range'
      write(LER,*)'                         -min:-max'
      write(LER,*)'              change - change one value to another'
      write(LER,*)'                         -old --> -new'
      write(LER,*)'              scale  - scaling only'
      write(LER,*)' '
      write(LER,*)'-S[]    -- Scalars for inputs                  (1.0)'
      write(LER,*)'             Repeat as many times as needed for'
      write(LER,*)'             multiple inputs.'
      write(LER,*)' '
      write(LER,*)'-B[]    -- Bias for inputs                     (0.0)'
      write(LER,*)'             Repeat as many times as needed for'
      write(LER,*)'             multiple inputs.'
      write(LER,*)' '
      write(LER,*)'-eps[]  -- Added to all files when mode.eq.div (0.0)'
      write(LER,*)' '
      write(LER,*)'-min[]  -- For mode=clip, clip to -min    (-1.0e+37)'
      write(LER,*)'-max[]  -- For mode=clip, clip to -max    (+1.0e+37)'
      write(LER,*)' '
      write(LER,*)'-old[]  -- For mode=change.               (-1.0e+37)'
      write(LER,*)'-new[]  --    change -old to -new         (-1.0e+37)'
      write(LER,*)' '
      write(LER,*)'-emask[] -- value for masked samples       (no mask)'
      write(LER,*)'            omit if there is no embedded mask'
      write(LER,*)' '
      write(LER,*)'-V      -- Verbose printout                (.false.)'
      write(LER,*)' '
      write(LER,*)'Usage:'
      write(LER,*)'  uspmath -N[] -N[] ... -N[n] -O[]'
      write(LER,*)'          -S[] -S[] ... -S[n] -B[] -B[] ... -B[n]'
      write(LER,*)'          -mode[] -eps[] -min[] -max[]'
      write(LER,*)'          -old[] -new[] -emask[] -V'
      write(LER,*)' '
      write(LER,*)'===================================================='
      
      return
      end


c...|....1....|....2....|....3....|....4....|....5....|....6....|....7..
c Pick up command line arguments 
      subroutine cmdln
     :      (ntap, nfiles, otap, mode, scalar, bias, epsilon,
     :       minval,maxval, oldval,newval, emask,lmask, likp,
     :       name, verbose)

#include <f77/iounit.h>

      integer   nfiles
      real      scalar(*),bias(*),epsilon,emask
      real      minval,maxval, oldval,newval
      character*255 ntap(*), otap, ftap
      character     name*(*), mode*(*)
      logical   verbose, lmask, likp

      integer   argis, findarg, lenthr
      integer   ifile, lufil, ier, ii, len1
      character cardin*255
      logical   found_file, found_param

c First see if we're running in ikp
c   This argument is set in scripts called from XIKP. It tells
c   uspmath the maximum number of pipes to expect.
c   ** Command line users shouldn't need it. **
      call argi4('-ikp',nfiles,-1,-1)
      likp = .false.
      if (nfiles .gt. 1) likp=.true.

      if (nfiles .le. 1) then

c Next try to retrieve a file with the name of all inputs
        call argstr('-F', ftap, ' ', ' ') 

c Now retrieve all the names on input
        if (ftap(1:1) .eq. ' ') then

          nfiles = 1
          found_file = .true.
          ifile = 0
          do while (found_file)
            ifile = ifile + 1
            call argstr('-N', ntap(ifile), ' ', ' ') 
            call argr4('-S', scalar(ifile), 1.0, 1.0)
            call argr4('-B', bias(ifile), 0.0, 0.0)
            if (ntap(ifile)(1:1).eq.' ') then
              found_file=.false.
              if (ifile.gt.1) nfiles = ifile-1
            endif
          enddo

        else

          call alloclun (lufil)
          open (unit=lufil, file=ftap, status='old', iostat=ier)
          if(ier .gt. 0) then
            write(LERR,*)name,': Failed to open file of input names'
            write(LERR,*)name,':    ',ftap(1:lenthr(ftap))
            write(LERR,*)name,': FATAL'
            write(LER,*)name,': Failed to open file of input names'
            write(LER,*)name,':    ',ftap(1:lenthr(ftap))
            write(LER,*)name,': FATAL'
            stop
          endif
          rewind lufil

          nfiles = 1
          found_file = .true.
          ifile = 0
          ier = 0
          do while (ier.eq.0 .and. found_file)
            ifile = ifile + 1

            do ii = 1,255
              cardin(ii:ii) = ' '
            enddo

            read (unit=lufil,fmt='(a255)',iostat=ier) cardin

            if (ier .gt. 0) then
              write(LERR,*)name,': Read error on ftap at line ',ifile
              write(LERR,*)'FATAL'
              write(LER ,*)name,': Read error on ftap at line ',ifile
              write(LER ,*)'FATAL'
              close (unit=LERR)
              close (unit=lufil)
              stop

            elseif (ier .lt. 0) then
              write(LERR,*)name,': End of ftap detected at file ',ifile
              write(LER ,*)name,': End of ftap detected at file ',ifile
              nfiles = ifile-1

            else
              len1 = lenthr(cardin)
              ntap(ifile) = cardin(1:len1)

              if (ntap(ifile)(1:1).eq.' ') then
                found_file=.false.
                if (ifile.gt.1) nfiles = ifile-1
              else
                call parse ('-S', cardin, value, found_param)
                if (found_param .eq. 0) value = 1.0
                scalar(ifile) = value
                call parse ('-B', cardin, value, found_param)
                if (found_param .eq. 0) value = 0.0
                bias(ifile) = value
              endif
            endif

          enddo

          close (unit=lufil)

        endif
      endif

c Allow first of many named inputs to be a pipe
      if (ntap(1)(1:6).eq.'stdin:') ntap(1)=' '

      call argstr('-O', otap, ' ', ' ') 
      call argstr('-mode', mode, 'sum', 'sum') 

c Check for irregularities

      if (mode(1:3).eq.'sum' .and. nifles.eq.1) then
        write(LER,*)name,': Parameter error. When mode.eq.sum'
        write(LER,*)name,': multiple files are needed on input'
        write(LER,*)name,': Fatal'
        stop
      endif

      if (mode(1:4).eq.'mult' .and. nfiles.eq.1) then
        write(LER,*)name,': Parameter error. When mode.eq.mult'
        write(LER,*)name,': multiple files are needed on input'
        write(LER,*)name,': Fatal'
        stop
      endif

      if (mode(1:3).eq.'div') then
        if(nfiles.ne.1) then
          call argr4('-eps', epsilon, 0.0, 0.0 )
          if (epsilon.eq.0.0) then
            write(LER,*)name,': Warning - epsilon is 0.0 and division'
            write(LER,*)name,':           by zero is possible.'
            write(LER,*)name,': To avoid this use -eps on the command.'
            write(LER,*)name,': Epsilon is added to the bias for each'
            write(LER,*)name,': input file.'
          else
            do ifile=1,nfiles
              bias(ifile)=bias(ifile)+epsilon
            enddo
          endif
        else
          write(LER,*)name,': Parameter error. When mode.eq.div'
          write(LER,*)name,': multiple files are needed on input'
          write(LER,*)name,': Fatal'
          stop
        endif
      endif

      if (mode(1:3).eq.'min' .and. nfiles.eq.1) then
        write(LER,*)name,': Parameter error. When mode.eq.min'
        write(LER,*)name,': multiple files are needed on input'
        write(LER,*)name,': Fatal'
        stop
      endif

      if (mode(1:3).eq.'max' .and. nfiles.eq.1) then
        write(LER,*)name,': Parameter error. When mode.eq.max'
        write(LER,*)name,': multiple files are needed on input'
        write(LER,*)name,': Fatal'
        stop
      endif

      if (mode(1:5).eq.'union' .and. nfiles.eq.1) then
        write(LER,*)name,': Parameter error. When mode.eq.union'
        write(LER,*)name,': multiple files are needed on input'
        write(LER,*)name,': Fatal'
        stop
      endif

      if (mode(1:4).eq.'clip') then
        if(nfiles.eq.1) then
          call argr4('-min', minval, -1.0e+37, -1.0e+37 )
          call argr4('-max', maxval,  1.0e+37,  1.0e+37 )
        else
          write(LER,*)name,': Parameter error. When mode.eq.clip'
          write(LER,*)name,': only one file is allowed on input.'
          write(LER,*)name,': Fatal'
          stop
        endif
      endif

      if (mode(1:6).eq.'change') then
        if(nfiles.eq.1) then
          call argr4('-old', oldval, -1.0e+37, -1.0e+37 )
          call argr4('-new', newval, -1.0e+37, -1.0e+37 )
        else
          write(LER,*)name,': Parameter error. When mode.eq.change'
          write(LER,*)name,': only one file is allowed on input.'
          write(LER,*)name,': Fatal'
          stop
        endif
      endif

      if (mode(1:5).eq.'scale') then
        if(nfiles.ne.1) then
          write(LER,*)name,': Parameter error. When mode.eq.scale'
          write(LER,*)name,': only one file is allowed on input.'
          write(LER,*)name,': Fatal'
          stop
        endif
      endif

      verbose= .false.
      verbose= (argis('-V').gt.0)

      lmask = .false.
      if (findarg('-emask').gt.0) then
        lmask = .true.
        call argr4('-emask', emask, -1.0e+37, -1.0e+37 )
      endif

c Some arguments are only parsed when the relevant mode is set
c XIKP needs defaults for just about everything so such arguments
c will be on all command lines set up by XIKP. When not needed,
c these will look like extra arguments and be reported by xtrarg().
c This little bit alleviates anxiety that might be caused by the
c xtrarg() reports
      if (likp .or. nfiles.eq.1) then
        call argr4('-eps', epsilon, epsilon, epsilon )
        call argr4('-min', minval, minval, minval )
        call argr4('-max', maxval, maxval, maxval )
        call argr4('-old', oldval, oldval, oldval )
        call argr4('-new', newval, newval, newval )
      endif

c Check for extraneous arguments and report if found.
c All manner of legitimate oversight should have been taken care of
c already, so if there are any extra args, kill the job.
      call xtrarg (name, ler,  .FALSE., .FALSE.)
      call xtrarg (name, lerr, .FALSE., .TRUE.)

      return
      end


c...|....1....|....2....|....3....|....4....|....5....|....6....|....7..
c Verbal printout of pertinent program particulars
      subroutine verbal
     :        (ntap, nfiles, otap, mode, scalar,bias,epsilon,
     :         n1,n2,n3, minval,maxval, oldval,newval,
     :         emask,lmask, verbose)

#include <f77/iounit.h>

      character*255 ntap(nfiles)
      character     otap*(*),mode*(*)
      integer    n1,n2,n3, nfiles
      real       scalar(nfiles),bias(nfiles),epsilon,emask
      real       minval,maxval, oldval,newval
      logical    verbose,lmask
      integer    nblen,if1

      write(LERR,*)' '
      write(LERR,*)'File names for input and output'
      write(LERR,*)' '
      write(LERR,*)'   Input data set names:'
      if1 = 1
      if (ntap(1)(1:1).eq.' ') then
        write(LERR,*)'      stdin:'
        if1 = 2
      endif
      do ifile=if1,nfiles
        write(LERR,*)'      ',ntap(ifile)(1:nblen(ntap(ifile)))
      enddo
      write(LERR,*)' '
      write(LERR,*)'   Output data set name:'
      if (otap(1:1).eq.' ') then
        write(LERR,*)'      stdout:'
      else
        write(LERR,*)'      ',otap(1:nblen(otap))
      endif
      write(LERR,*)' '
      write(LERR,*)'   Data sizes:'
      write(LERR,*)'      Number of samples   = ',n1
      write(LERR,*)'      Number of traces    = ',n2
      write(LERR,*)'      Number of records   = ',n3
      write(LERR,*)' '
      write(LERR,*)' '
      write(LERR,*)'Other command line parameters'
      write(LERR,*)' '
      write(LERR,*)'   Data scalars and biases:'
      do ifile=1,nfiles
        write(LERR,*)'       ',ifile,scalar(ifile),bias(ifile)
      enddo
      write(LERR,*)' '
      write(LERR,*)'   Mode of operation = ',mode(1:nblen(mode))
      if(mode(1:4).eq.'clip') then
        write(LERR,*)'       minval        = ',minval
        write(LERR,*)'       maxval        = ',maxval
        write(LERR,*)' '
      endif
      if(mode(1:3).eq.'div') then
        write(LERR,*)'       epsilon       = ',epsilon
        write(LERR,*)' '
      endif
      if(mode(1:4).eq.'change') then
        write(LERR,*)'       oldval        = ',oldval
        write(LERR,*)'       newval        = ',newval
        write(LERR,*)' '
      endif

      if (lmask) then
        write(LERR,*)'   Embedded mask value ',emask
        write(LERR,*)' '
      endif
      if (verbose) then
        write(LERR,*)'   Verbose printout requested'
        write(LERR,*)' '
      endif
      write(LERR,*)' '

      return
      end
