C***********************************************************************
C                 copyright 2001, Amoco Production Company             *
C                             All Rights Reserved                      *
C                     an affiliate of BP America Inc.                  *
C***********************************************************************
C***********************************************************************
c examines a seismic record for max coherency & slowness as a function
c of position within the record then outputs 2 records:
c 1) the slowness values
c 2) the max coherency values
 
c It is assumed that the development location has afp - Amoco Fortran
c Preprocessor
c
c
c**********************************************************************c
c
c     declare variables
c
c-----
c    get machine dependent parameters
c    these includes are picked up and expanded into the code by afp
c    and are found in ~usp/include/f77
 
#include <save_defs.h>
#include <f77/iounit.h>
#include <f77/lhdrsz.h>
#include <f77/sisdef.h>
c-----
 
c-----
c    SZLNHD is a value obtained from lhdrsz.h
c    The 3 vectors below are equivalenced and are
c    to access the trace header entries (whatever
c    they may be)
c-----
      integer     itr ( SZLNHD )
      integer     nsamp, nsi, ntrc, nrec, iform, obytes
      integer     luin , luouts, luoutc, lbytes, nbytes, lbyout
      integer     irs,ire,ns,ne
 
c------
c  dynamic memory allocation for big arrays, eg whole records
      integer     itrhdr
      real        win, data, slow, coh, wt, wx
      pointer     (wkadri, itrhdr  (1))
      pointer     (wkadrw, win     (1))
      pointer     (wkadrd, data    (1))
      pointer     (wkadrs, slow    (1))
      pointer     (wkadrc, coh     (1))
      pointer     (wkadrt, wt      (1))
      pointer     (wkadrx, wx      (1))
c------
 
c------
c    this is necessary for buliding printout files
#include <f77/pid.h>
c------
      integer     recnum, trcnum
      integer     srcloc, recind, dphind, dstsgn, stacor
      real        smin,smax,dx,vel
c-----
c    SZSMPM is a value obtained from lhdrsz.h
c-----
      real        tri ( SZLNHD )
      character   ntap * 255, otaps * 255, otapc * 255, name*6
      logical     verbos, query, heap, heapi
      integer     argis, pipe
 
      data lbytes / 0 /
      data nbytes / 0 /
      data name/'VELCOH'/
      data pipe /3/
 
      pi  = 3.14159265
      pi2 = 2. * pi
c-----
c     read program parameters from command line card image file
c-----
      query = ( argis ( '-?' ) .gt. 0 .or. argis ( '-h') .gt. 0)
      if ( query )then
            call help()
            stop
      endif
 
c-----
c     open printout files
c     this consists of a 2-character identifier, the process id number,
c     and the child process id number:
c     XY.nnnnn.mmmmm
c     this should be unique even for multiple occurences of the same
c     process in a pipeline
c-----
#include <f77/open.h>
 
      call gcmdln(ntap,otaps,otapc,ns,ne,irs,ire,verbos,
     1            vel,dmin,dmax,nsearch,dx,nrows,ncols)

c-----
c     get logical unit numbers for input and output of seismic data
 
c     input values are strings ntap & otap (i.e. names of files), and the
c     read 'r' or write 'w' designations of these files, and the default
c     logical unit numbers if ntap & otap are blank strings, i.e. ntap = " "
c     in which case:
c     0 = default stdin
c     1 = default stdout
c     Note: default values other that 0 & 1 may be used
 
c     output values are the logical unit numbers accessing these disk files. if
c     these values are less than 0 it means there was a fatal error in trying to
c     open these files
c-----
      call getln(luin  , ntap ,'r', 0)
      call getln(luouts, otaps,'w', 1)

      if (otapc(1:1) .ne. ' ') then
          call getln(luoutc, otapc, 'w',-1)
      else
          write(LERR,*)'velcoh assumed to be running inside IKP'
          call sisfdfit (luoutc, pipe)
      endif
      if  (luoutc .lt. 0)   then
           write(LERR,*)'velcoh error: velocity file -OC not accessible'
           call ccexit (666)
      endif
 
c-----
c     read line header of inputa DSN (rtape reads data into vector "itr"
c     lbytes is the number of bytes actually read
c-----
      call rtape  ( luin, itr, lbytes)
      if(lbytes .eq. 0) then
         write(LOT,*)'VELCOH: no header read from unit ',luin
         write(LOT,*)'FATAL'
         call ccexit (666)
      endif
 
c------
c     save certain parameters
 
c     For trace header values we take mnemonics and build a
c     set of pointers 
c     TRACEHEADER is a value in the include file <sisdef.h> that 
c     refers to the trace header

      call savelu('TrcNum',ifmt_TrcNum,l_TrcNum,ln_TrcNum,TRACEHEADER)
      call savelu('RecNum',ifmt_RecNum,l_RecNum,ln_RecNum,TRACEHEADER)
      call savelu('SrcLoc',ifmt_SrcLoc,l_SrcLoc,ln_SrcLoc,TRACEHEADER)
      call savelu('RecInd',ifmt_RecInd,l_RecInd,ln_RecInd,TRACEHEADER)
      call savelu('DphInd',ifmt_DphInd,l_DphInd,ln_DphInd,TRACEHEADER)
      call savelu('DstSgn',ifmt_DstSgn,l_DstSgn,ln_DstSgn,TRACEHEADER)
      call savelu('StaCor',ifmt_StaCor,l_StaCor,ln_StaCor,TRACEHEADER)

      call saver(itr, 'NumSmp', nsamp, LINHED)
      call saver(itr, 'SmpInt', nsi  , LINHED)
      call saver(itr, 'NumTrc', ntrc , LINHED)
      call saver(itr, 'NumRec', nrec , LINHED)
      call saver(itr, 'Format', iform, LINHED)
      call saver(itr, 'UnitSc', unitsc, LINHED)
      if (unitsc .eq. 0.0) then
          write(LERR,*)'********************************************'
          write(LERR,*)'WARNING: sample unit scaler in LH = ',unitsc
          write(LERR,*)'         will set to .001 (millisec default)'
          write(LERR,*)'********************************************'
          unitsc = .001
          call savew(itr, 'UnitSc', unitsc, LINHED)
      endif
 
c------
c     hlhprt prints out the historical line header of length lbytes AND
 
c     hlhprt takes "name", in this case 4 characters long and stuffs this
c     into the modified historical line header and returns the NEW length
c     of the line header in lbytes
c------
      call hlhprt (itr, lbytes, name, 6, LERR)
 
c-----
c     ensure that command line values are compatible with data set
c     (i.e. start/end traces; start/end records).  This guards against
c     zero start values or values that are greater than those specified
c     in the line header
c-----
      call cmdchk(ns,ne,irs,ire,ntrc,nrec)
 
c---------------------------------------------------
c  malloc only space we're going to use
      heapi = .true.
      heap  = .true.
 
c--------------------------
c  note: these don't
c  have to be the same size
 
 
      if (mod(ncols,2) .eq. 0) ncols = ncols + 1
      if (mod(nrows,2) .eq. 0) nrows = nrows + 1
      nrows2  = nrows / 2
      ncols2  = ncols / 2
      nrows21 = nrows2 - 1
      ncols21 = ncols2 - 1

      itemi = ntrc * ITRWRD          * SZSMPD
      itemw = (nrows+1) * (ncols+1)  * SZSMPD
      itemd = (ntrc + ncols)  * (nsamp + nrows) * SZSMPD
      items = (ntrc+1) * (nsamp+1)   * SZSMPD
      itemr = (2 * nrows + 1)        * SZSMPD
      itemt = max (nrows, ncols)     * SZSMPD
 
c  note also SZSMPD is the native
c  size of a float or int in bytes
c--------------------------
 
c--------
c  galloc - general allocation (machine independent since it uses C
c  malloc internally
c  inputs to galloc are pointer, number of bytes to allocate
c  outputs are error codes:
c     errcod = 1  (allocation succeeded)
c     errcod = 0  (allocation failed)
c--------
 
      call galloc (wkadri, itemi, errcdi, aborti)
      call galloc (wkadrw, itemw, errcdw, abortw)
      call galloc (wkadrd, itemd, errcd1, abort1)
      call galloc (wkadrc, items, errcd2, abort2)
      call galloc (wkadrs, items, errcd3, abort3)
      call galloc (wkadrt, itemt, errcd4, abort4)
      call galloc (wkadrx, itemt, errcd5, abort5)
 
      if (errcdi .ne. 0.) heapi = .false.
      if (errcdw .ne. 0.) heap  = .false.
      if (errcd1 .ne. 0.) heap  = .false.
      if (errcd2 .ne. 0.) heap  = .false.
      if (errcd3 .ne. 0.) heap  = .false.
      if (errcd4 .ne. 0.) heap  = .false.
      if (errcd5 .ne. 0.) heap  = .false.
 
      if (.not. heap .or. .not. heapi) then
         write(LERR,*)' '
         write(LERR,*)'Unable to allocate workspace:'
         write(LERR,*) itemi,'  bytes'
         write(LERR,*) itemw,'  bytes'
         write(LERR,*) itemd,'  bytes'
         write(LERR,*) items,'  bytes'
         write(LERR,*) items,'  bytes'
         write(LERR,*) itemt,'  bytes'
         write(LERR,*) itemt,'  bytes'
         write(LERR,*)' '
         go to 999
      else
         write(LERR,*)' '
         write(LERR,*)'Allocating workspace:'
         write(LERR,*) itemi,'  bytes'
         write(LERR,*) itemw,'  bytes'
         write(LERR,*) itemd,'  bytes'
         write(LERR,*) items,'  bytes'
         write(LERR,*) items,'  bytes'
         write(LERR,*) itemt,'  bytes'
         write(LERR,*) itemt,'  bytes'
         write(LERR,*)' '
      endif
c---------------------------------------------------
 
 
c-----
c     modify line header to reflect actual number of traces output
c-----
      nrecc = ire - irs + 1
      call savew(itr, 'NumRec', nrecc, LINHED)
      jtr   = ne - ns + 1
      call savew(itr, 'NumTrc', jtr  , LINHED)
 
c----------------------
c  number output bytes
      obytes = SZTRHD + nsamp * SZSMPD
 
c----------------------
c  inject command line into
c  historical LH:
c  inputs are current line header in vector "itr" and length lbytes (bytes)
c  outputs are line header modified by insertion of the command line into
c  the historical LH and the modified length of the new LH (lbyout)
 
      call savhlh(itr,lbytes,lbyout)
c----------------------
 
c------
c     write to unit number luout lbyout bytes contained in vector itr
c------
      call wrtape ( luouts, itr, lbyout                 )
      call wrtape ( luoutc, itr, lbyout                 )

c--------------------------------------------------
c  compute sample interval in secs
c  take care of micro secs if necessary
 
      dt = real (nsi) * unitsc
 
      smin = sin (dmin  * pi / 180.) / vel
      smax = sin (dmax * pi / 180. ) / vel

      call vfill (1.0, wt, 1, nrows)
      call vfill (1.0, wx, 1, ncols)

      do  i = 1, nrows21
          wt(i)  = .5 * (1. - cos ( pi * i /nrows21) )
          ii = nrows - i + 1
          wt(ii) = wt(i)
      enddo
      do  i = 1, ncols21
          wx(i)  = .5 * (1. - cos ( pi * i /ncols21) )
          ii = ncols - i + 1
          wx(ii) = wx(i)
      enddo
c-----
c     verbose output of all pertinent information before
c     processing begins
c-----
      if( verbos ) then
            call verbal(nsamp, nsi, ntrc, nrec, iform,
     1                  ntap,otaps,otapc,dt,irs,ire,
     2                  vel,dmin,dmax,nsearch,dx,nrows,ncols,
     3                  wt,wx)
      end if
 
c--------------------------------------------------
c-----
c     BEGIN PROCESSING
c     read trace, do terrible things to data, write to output file
c-----
c-----
c     skip unwanted records
c-----
      call recskp(1,irs-1,luin,ntrc,itr)
 
c-----
c     process desired trace records
c-----
      do 1000 jj = irs, ire
 
            ic = 0
            call vclr (data, 1, (ntrc+ncols)*(nsamp+nrows))
            do 1001  kk = 1, ntrc
 
                  nbytes = 0
                  call rtape( luin, itr, nbytes)
c------
c     if end of data encountered (nbytes=0) then bail out
c     Note:  if you're processing records you might really want
c     to branch to the processing part rather than bailing out
c------
                  if(nbytes .eq. 0) then
                     write(LERR,*)'End of file on input:'
                     write(LERR,*)'  rec= ',jj,'  trace= ',kk
                     go to 999
                  endif
                  call vmov (itr(ITHWP1), 1, tri, 1, nsamp)
 
c------
c     use previously derived pointers to trace header values

                  call saver2( itr, ifmt_RecNum, l_RecNum, ln_RecNum, 
     :                 recnum, TRACEHEADER)
                  call saver2( itr, ifmt_TrcNum, l_TrcNum, ln_TrcNum, 
     :                 trcnum, TRACEHEADER)
                  call saver2( itr, ifmt_SrcLoc, l_SrcLoc, ln_SrcLoc, 
     :                 srcloc, TRACEHEADER)
                  call saver2( itr, ifmt_RecInd, l_RecInd, ln_RecInd, 
     :                 recind, TRACEHEADER)
                  call saver2( itr, ifmt_DphInd, l_DphInd, ln_DphInd, 
     :                 dphind, TRACEHEADER)
                  call saver2( itr, ifmt_DstSgn, l_DstSgn, ln_DstSgn, 
     :                 dstsgn, TRACEHEADER)
                  call saver2( itr, ifmt_StaCor, l_StaCor, ln_StaCor, 
     :                 stacor, TRACEHEADER)

                  if (stacor .eq. 30000) then
                     call vclr (tri,1,nsamp)
                  endif
 
c----------------------
c  pack data into array
                  ic = ic + 1
c                 istrc = (ic+nrows2) * nsamp
                  istrc = (ic+nrows2) * (nsamp+nrows)
                  ishdr = (ic-1) * ITRWRD
                  call vmov (tri,1, data  (istrc+1),1, nsamp)
                  call vmov (itr,1, itrhdr(ishdr+1),1,ITRWRD)
 
1001        continue
 
 
c-----------------------
c  here's the meat...
 
            call  movwin (dt, dx, smax, smin, sstep, 
     1                    nrows, ncols, nrows2, ncols2,
     2                    nsearch, vel,
     3                    nsamp, ntrc, wt, wx,
     4                    win, data, slow, coh, verbos)

 
c-----------------------
 
c---------------------
c  extract traces from
c  output array and
c  write output data
            do 1002 kk = 1, jtr
 
                  istrc = (kk-1) * nsamp
                  ishdr = (kk-1) * ITRWRD
                  call vmov (slow  (istrc+1),1,itr(ITHWP1),1, nsamp)
                  call vmov (itrhdr(ishdr+1),1,itr,1,ITRWRD)
                  call wrtape (luouts, itr, obytes)

                  call vmov (coh   (istrc+1),1,itr(ITHWP1),1, nsamp)
                  call wrtape (luoutc, itr, obytes)
 
 
 1002             continue
 
            if (verbos)
     1      write(LERR,*)'Record ',recnum,'  processed'
 
 1000       continue
 
  999 continue
 
c-----
c     close data files
c     flush data left
c     in output buffer
c     NOTE: if the output buffer is not closed you can sometimes end up
c     with missing data
c-----
          call lbclos ( luin )
          call lbclos ( luouts )
      if (luoutc .gt. 0)
     1    call lbclos ( luoutc )
 
            write(LERR,*)'end of prgm, processed',nrec,' record(s)',
     :               ' with ',ntrc, ' traces'
      end
 
C***********************************************************************
      subroutine help
#include <f77/iounit.h>
         write(LER,*)
     :'***************************************************************'
        write(LER,*) ' '
        write(LER,*)
     :'velcoh finds the slownesses (dips) with max coherency'
        write(LER,*)
     :'       (semblance) and outputs these 2 data sets'
        write(LER,*)
     :'see manual pages for details ( online by typing uman prg )'
        write(LER,*)' '
        write(LER,*)
     :'execute prgm by typing prgm and the of program parameters'
        write(LER,*)
     :'note that each parameter is proceeded by -a where "a" is '
        write(LER,*)
     :'a character(s) corresponding to some parameter.'
        write(LER,*)
     :'users enter the following parameters, or use the default values'
        write(LER,*)' '
        write(LER,*)
     :' -N [ntap]    (stdin)          : input data file name'
        write(LER,*)
     :' -OS [otaps]  (stdout)         : output slowness or angle'
        write(LER,*)
     :' -OC [otapc]  (none)           : output coherency'
        write(LER,*)
     :' -rs[irs]     (default = first)    : start record number'
        write(LER,*)
     :' -re[ire]     (default = last)     : end record number'
        write(LER,*) ' '
        write(LER,*)
     :' -v[vel]   (default = none)  :  surface (shot/receiver) velocity'
        write(LER,*)
     :' -dx[dist] (default = none)  :  trace to trace distance'
        write(LER,*)
     :' -ds[dmin] (default = -90)   :  min dip angle'  
        write(LER,*)
     :' -de[dmax] (default = +90)   :  max dip angle'  
        write(LER,*)
     :' -dn[nsearch] (default = 100):  number dip angle scans'
        write(LER,*)
     :' -wt[nrows] (default = none) :  # points in time scan window'
        write(LER,*)
     :' -wx[ncols] (default = none) :  # points in space scan window'
        write(LER,*) ' '
        write(LER,*)
     :' -V  include on command line if verbose printout is desired'
        write(LER,*)
     :'usage:   velcoh -N[ntap] -O[otap] -rs[irs] -re[ire] [-V -S]'
        write(LER,*)
     :'                 -v[] -dx[] -ds[] -de[] -dn[] -wt[] -wx[]'
        write(LER,*) ' '
        write(LER,*)
     :'***************************************************************'
      return
      end
 
C***********************************************************************
      subroutine gcmdln(ntap,otaps,otapc,ns,ne,irs,ire,verbos,
     1                  vel,dmin,dmax,nsearch,dx,nrows,ncols)
c-----
c     get command arguments
c
c     ntap  - C*100    input file name
c     otap  - C*100    output file name
c     vel   - R*4      design velocity
c     ns    - I*4      starting trace index
c     ne    - I*4      ending trace index
c     irs   - I*4      starting record index
c     ire   - I*4      ending record index
c     verbos  L        verbose output or not
c-----
#include <f77/iounit.h>
      character   ntap*(*), otaps*(*), otapc*(*)
      integer     ns, ne, irs, ire
      real        vel
      logical     verbos
      integer     argis
 
c-------
c     import values from the command line using keys, e.g. -N
c     to which are immediately attached the users values.
 
c     For example program prgm might be invoked in the following way:
 
c     prgm  -Nxyz -Oabc
 
c     in which case xyz is a string (the name of the input data set)
c     which will be imported into prgm and associated with the variable
c     "ntap"
 
c     see manual pages on the argument handler routines
c     for the meanings of these functions.  Briefly though
c     we can import from the command line strings, integers, reals,
c     double precision reals, and check the existence of command line
c     keys.  The last 2 fields are the value to be assigned the variable
c     (1) if ONLY the key is present (no value attached to it), or
c     (2) if NO key & no value are present
c-------
            call argstr( '-N' , ntap,  ' ', ' ' )
            call argstr( '-OS', otaps, ' ', ' ' )
            call argstr( '-OC', otapc, ' ', ' ' )
            call argi4 ( '-rs', irs ,   0  ,  0    )
            call argi4 ( '-re', ire ,   0  ,  0    )
            call argr4 ( '-v',   vel,      0.,      0. )
            call argr4 ( '-ds', dmin,    -90.,    -90. )
            call argr4 ( '-de', dmax,    +90.,    +90. )
            call argi4 ( '-dn', nsearch ,  21  , 21    )
            call argr4 ( '-dx',   dx ,     0.,      0. )
            call argi4 ( '-wt',nrows,   10  , 10    )
            call argi4 ( '-wx',ncols,   10  , 10    )
            verbos =   (argis('-V') .gt. 0)
 
            if (mod(nsearch,2) .eq. 0) nsearch = nsearch + 1
            if (vel .eq. 0.0) then
               write(LERR,*)'Must enter nonzero velocity -- FATAL'
               write(LERR,*)'Re-run with -v[] on cmd line'
               stop 911
            endif
            if (dx .eq. 0.0) then
               write(LERR,*)'Must enter nonzero trace spacing -- FATAL'
               write(LERR,*)'Re-run with -dx[] on cmd line'
               stop 911
            endif
            if (nrows .eq. 0) then
               write(LERR,*)'Must enter nonzero # pts in time wind'
               write(LERR,*)'Re-run with -wt[] on cmd line'
               stop 911
            endif
            if (ncols .eq. 0) then
               write(LERR,*)'Must enter nonzero # pts in space wind'
               write(LERR,*)'Re-run with -wx[] on cmd line'
               stop 911
            endif
c-------
c     Note:  argis() = 1 if the key is present
c            argis() = 0 if it isn't

            ns = 0
            ne = 0
 
      return
      end
 
C***********************************************************************
      subroutine verbal(nsamp, nsi, ntrc, nrec, iform,
     1                  ntap,otaps,otapc,dt,irs,ire,
     2                  vel,dmin,dmax,nsearch,dx,nrows,ncols,
     3                  wt,wx)

c-----
c     verbose output of processing parameters
c
c     nsamp - I*4     number of samples in trace
c     vel   - R*4     design velocity
c     nsi   - I*4     sample interval in ms
c     ntrc  - I*4     traces per record
c     nrec  - I*4     number of records per line
c     iform - I*4     format of data
c     ntap  - C*100   input file name
c     otap  - C*100   output file name
c-----
#include <f77/iounit.h>
 
      integer     nsamp, nsi, ntrc, nrec
      integer     irs,ire,nsearch,nrows,ncols
      real        vel,dmin,dmax,dx,dt
      real        wt(*), wx(*)
      character   ntap*(*), otaps*(*), otapc*(*)
 
            write(LERR,*)' '
            write(LERR,*)' line header values after default check '
            write(LERR,*) ' # of samples/trace      =  ', nsamp
            write(LERR,*) ' sample interval (ms)    =  ', nsi
            write(LERR,*) ' sample interval (sec)   =  ', dt
            write(LERR,*) ' trace spacing           =  ', dx
            write(LERR,*) ' traces per record       =  ', ntrc
            write(LERR,*) ' records per line        =  ', nrec
            write(LERR,*) ' format of data          =  ', iform
            write(LERR,*) ' start record            =  ', irs
            write(LERR,*) ' end record              =  ', ire
            write(LERR,*) ' surface velocity        =  ', vel
            write(LERR,*) ' min dip angle           =  ', dmin
            write(LERR,*) ' max dip angle           =  ', dmax
            write(LERR,*) ' number of dips          =  ', nsearch
            write(LERR,*) ' number of rows in wind  =  ', nrows
            write(LERR,*) ' number of cols in wind  =  ', ncols
            write(LERR,*) ' input data set name     =  ', ntap
            write(LERR,*) ' output slowness         =  ', otaps
            write(LERR,*) ' output coherency        =  ', otapc
            write(LERR,*)' '
            write(LERR,*) 'Temporal weights'
            write(LERR,*) (wt(i), i = 1, nrows)
            write(LERR,*) 'Spatial weights'
            write(LERR,*) (wx(i), i = 1, ncols)
            write(LERR,*)' '
            write(LERR,*)' '

      return
      end
 
