C***********************************************************************
C                 copyright 2001, Amoco Production Company             *
C                             All Rights Reserved                      *
C                     an affiliate of BP America Inc.                  *
C***********************************************************************
C**********************************************************************C
C**********************************************************************C
C
C      PROGRAM MODULE VELSPEC: velocity spectra ( as sis files )
C
C**********************************************************************C
c      B.V. NGUYEN and R.B. HERRMANN                     APRIL 1986 SLU
C**********************************************************************C
C          THE MAIN PROGRAM CALL SUBROUTINE TVEL TO PERFORM HYPERBOLIC-
C      TRAJECTORY SCAN SEARCH FOR OPTIMUM STACKING VELOCITY (RMS VEL.)
C      AS A FUNCTION OF TWO-WAY ZERO-OFFSET TIME VIA A COHERENCY MEASURE
C      CALLED SEMBLANCE IN A CMP GATHER OF TRACES.
C      THE SURFACE REPRESENTING THE COHERENCY MEASURE AS A FUNCTION OF
C      TWO-WAY ZERO-OFFSET TIME AND TEST STACKING VELOCITY DEFINES A
C      VELOCITY SPECTRUM.
C      FOR SEMBLANCE REFERENCE:
C      TANER, M.T. and F. KOEHLER (1969) GEOPHYSICS,34,p.859-881.
C      FOR ENERGY-NORMALIZED CROSS-CORRELATION SUM REFERENCE:
C      HUBRAL, P. and T. KREY (LARNER, K. L., Ed.) (1980) SOCIETY OF
C      EXPLORATION GEOPHYSICISTS, 203pp.
C**********************************************************************C

#include <localsys.h>
#include <f77/iounit.h>
#include <f77/lhdrsz.h>
#include <f77/sisdef.h>
#include <save_defs.h>

c declare standard USP variables

      integer     itr ( SZLNHD )
      integer     str ( SZLNHD )
      integer     luin, nbytes, obytes, iform, lbytes
      integer     nsamp, nsi, ntrc, nrec, ns, ne
      integer     istop, istart, npt
      integer     argis

      real        start, stop, dt, UnitSC
      real        tri ( SZSMPM )

      character   name * 7, ntap * 255, otap * 255, fpik * 255

      logical     verbos

c declare program specific variables

      integer     irecz( SZLNHD ), itrcz( SZLNHD ), itz  ( SZLNHD )
      integer     nseg(SZLNHD), non_zero_samples(SZLNHD)
      integer     igt, nvel

      real        tmul, dmul
      real        dis( SZLNHD ), weight( SZLNHD )
      real        vello( SZLNHD ), velf ( SZLNHD ), tmp1 ( SZLNHD )
      real        velhi( SZLNHD ),   tf ( SZLNHD ), tmp2 ( SZLNHD )
      real        snum ( SZLNHD ), sden ( SZLNHD ), cn   ( SZLNHD )
      real        vmin,vmax

      logical     abslut, linear,slowness
      logical     heap, num, recpik, rev, logsc, nowt

c declare variables use with dynamic memory allocation

      real        traces, vels

      pointer     (wkaddr, traces(1))
      pointer     (wkvels, vels(1))

c initialize variables

      data name /'VELSPEC'/
      data  nbytes / 0 /
      data  lbytes / 0 /
c______________________________________________________________________
c     calculate trace header locations.
c______________________________________________________________________
      call savelu('TrcNum',ifmt_TrcNum,l_TrcNum,ln_TrcNum,TRACEHEADER)
      call savelu('RecNum',ifmt_RecNum,l_RecNum,ln_RecNum,TRACEHEADER)
      call savelu('SrcLoc',ifmt_SrcLoc,l_SrcLoc,ln_SrcLoc,TRACEHEADER)
      call savelu('RecInd',ifmt_RecInd,l_RecInd,ln_RecInd,TRACEHEADER)
      call savelu('DphInd',ifmt_DphInd,l_DphInd,ln_DphInd,TRACEHEADER)
      call savelu('LinInd',ifmt_LinInd,l_LinInd,ln_LinInd,TRACEHEADER)
      call savelu('DstSgn',ifmt_DstSgn,l_DstSgn,ln_DstSgn,TRACEHEADER)
      call savelu('DstUsg',ifmt_DstUsg,l_DstUsg,ln_DstUsg,TRACEHEADER)
      call savelu('StaCor',ifmt_StaCor,l_StaCor,ln_StaCor,TRACEHEADER)
      call savelu('SrPtXC',ifmt_SrPtXC,l_SrPtXC,ln_SrPtXC,TRACEHEADER)
      call savelu('SrPtYC',ifmt_SrPtYC,l_SrPtYC,ln_SrPtYC,TRACEHEADER)
      call savelu('RcPtXC',ifmt_RcPtXC,l_RcPtXC,ln_RcPtXC,TRACEHEADER)
      call savelu('RcPtYC',ifmt_RcPtYC,l_RcPtYC,ln_RcPtYC,TRACEHEADER)
      call savelu('CDPBCX',ifmt_CDPBCX,l_CDPBCX,ln_CDPBCX,TRACEHEADER)
      call savelu('CDPBCY',ifmt_CDPBCY,l_CDPBCY,ln_CDPBCY,TRACEHEADER)


C*********************************************************************C
c  get online help if necessary
C*********************************************************************C
      if ( argis ('-?') .gt. 0 .or. 
     :     argis ('-h') .gt. 0 .or. 
     :     argis ('-help') .gt. 0 ) then
         call help ()
         stop
      endif

C*********************************************************************C
c  open printout file
C*********************************************************************C
#include <f77/open.h>

C**********************************************************************C
C     READ PROGRAM PARAMETERS FROM COMMAND LINE
C**********************************************************************C
      call cmdln (ntap,otap,start,stop,dmul,tmul,igt,
     &     ns,ne,irs,ire,nvel,vmin,vmax,abslut,verbos,
     &     linear,slowness,thresh,fpik,logsc,nowt)

C**********************************************************************C
C     READ AND UPDATE LINE HEADER,
C     WRITE LINE HEADER, SAVE KEY PARAMETERS.
C**********************************************************************C
      call getln ( luin, ntap, 'r', 0 )
      call getln (luout, otap, 'w', 1 )

      lbytes = 0
      CALL RTAPE  ( LUIN, itr, LBYTES           )
      if(lbytes .eq. 0) then
         write(LERR,*)'VELSPEC: no header read on unit ',ntap
         write(LERR,*)'FATAL'
         write(LERR,*)'Check existence of file & rerun'
         stop
      endif

      CALL HLHprt ( ITR , LBYTES, name, 7, LERR         )

c-----------------------------------------
c  save key header values
#include <f77/saveh.h>

      write(LERR,*)' '
      write(LERR,*)' Values read from input data set lineheader'
      write(LERR,*)' # of Samples/Trace =  ',nsamp
      write(LERR,*)' Sample Interval    =  ',nsi
      write(LERR,*)' Traces per Record  =  ',ntrc
      write(LERR,*)' Records per Line   =  ',nrec
      write(LERR,*)' Format of Data     =  ',iform
      write(LERR,*)' Slowness**2 spectra?  ',slowness
      write(LERR,*)' Velocity spectra?     ',.not. slowness

C**********************************************************************C
C     CHECK DEFAULT VALUES AND SET PARAMETERS
C**********************************************************************C

      si = tmul * float(nsi) * UnitSC
      dt = si

      if(START.le.0.0)START = 0.0
      ttmax = (NSAMP-1) * si
      if(STOP.le.0.0 .or. STOP.ge. ttmax)STOP=ttmax
      ISTART =   INT   ( START / si ) + 1
      ISTOP  =   INT   ( STOP  / si ) + 1
     
      IF ( ISTART . LT. 1     .OR. ISTART.GT. NSAMP ) THEN
         ISTART = 1
      ENDIF
      
      IF ( ISTOP . LE. ISTART .OR. ISTOP .GT. NSAMP ) THEN
         ISTOP  = NSAMP
      ENDIF
      
      NPT    =       ISTOP - ISTART + 1
      
      if(ire.lt.1.or.ire.lt.irs.or.ire.gt.nrec)ire = nrec
      
      nrecc = ire - irs + 1
      if(ns.le.0 )ns = 1
      if(ne.le.0 .or. ne.gt.ntrc)ne=ntrc

      igt0 = 2 * igt + 1
      call vfill (1.0, weight, 1, igt0)

      if (igt0 .gt. 1 .AND. .not.nowt) then
         igt2 = igt / 2 
         do  i = 1, igt2
            ang = 3.14159265 * float(i-1)/float(igt2)
            weight (i) = .5 * (1. - cos ( ang ) )
            weight (igt0-i+1) = weight (i)
         enddo
      endif

C**********************************************************************C

      write(LERR,*)' '
      write(LERR,*)' Values read from file command line'
      write(LERR,*)' Start time in ms (default = 0 ms)    =  ',START
      write(LERR,*)' Stop time in ms (default= full trace) =  ',STOP
      write(LERR,*)' Start time in samples                =  ',ISTART
      write(LERR,*)' Stop time in samples                  =  ',ISTOP
      write(LERR,*)' Number of velocity inputs             =  ',nvel
      write(LERR,*)' Minimum velocity for plots            =  ',vmin
      write(LERR,*)' Maximum velocity for plots            =  ',vmax
      write(LERR,*)' Time gate integer                     =  ',igt
      write(LERR,*)' Turn off cosine weighting?            =  ',nowt
      write(LERR,*)' Distance multiplied by a factor       =  ',dmul
      write(LERR,*)' Time multiplied by a factor           =  ',tmul
      write(LERR,*)' First record to process               =  ',irs
      write(LERR,*)' Last record to process                =  ',ire
      write(LERR,*)' Starting trace number                 =  ',ns
      write(LERR,*)' Ending   trace number                 =  ',ne
      if (linear) then
         write(LERR,*)' Using linear velocity analysis'
      else
         write(LERR,*)' Using hyperbolic velocity analysis'
      endif
      write(LERR,*)' '
      write(LERR,*)' Weight applied to time gate:'
      write(LERR,*)(weight(ii),ii=1,igt0)
      write(LERR,*)' '

C**********************************************************************C
c  malloc only space we're going to use

      heap = .true.

      items = ntrc * (nsamp)
      itemv = nvel * (nsamp)
      call galloc (wkaddr, items*SZSMPD, errcod, abort)
      if (errcod .ne. 0) heap = .false.
      call galloc (wkvels, itemv*SZSMPD, errcod, abort)
      if (errcod .ne. 0) heap = .false.

      if (.not. heap) then
         write(LERR,*)' '
         write(LERR,*)'Unable to allocate workspace:'
         write(LERR,*) items*SZSMPD,'  bytes'
         write(LERR,*) itemv*SZSMPD,'  bytes'
         write(LERR,*)' '
         go to 999
      else
         write(LERR,*)' '
         write(LERR,*)'Allocating workspace:'
         write(LERR,*) items*SZSMPD,'  bytes'
         write(LERR,*) itemv*SZSMPD,'  bytes'
         write(LERR,*)' '
      endif

c
c  if no fairway picks the ignore the pick read stuff
c

      if (fpik(1:1) .eq. ' ') then
         write(LERR,*)'No velocity Fairway'
         call vfill (-999999., vello, 1, nsamp)
         call vfill (+999999., velhi, 1, nsamp)
         go to 111
      endif

c----
c find number of segments in velocity fairway pik file:
c  if there are 2 then get both segments and from time-vel
c  function build 2 velocity traces vello & velhi
c----

      nsi = si
      num = .true.

      call  pikrd ( nblk, nseg, iunit, iseg,nsi,itrs,itre,
     1     nsmp,recpik,irecz,itrcz,itz,num,LUDISK,
     2     unit1, unit2)
 
      if (nblk .ne. 2) then
         write(LERR,*)'Velocity fairway pick file does not contain'
         write(LERR,*)'required 2 pick segments (for low and high'
         write(LERR,*)'velocity limits)'
         write(LERR,*)'Ignoring fairway'
         call vfill (-999999., vello, 1, nsamp)
         call vfill (+999999., velhi, 1, nsamp)
         go to 111
      endif

      if (unit2 .eq. 1.0) then
         dvel = (vmax-vmin)/float(nvel-1)
      else
         dvel = 1.0
      endif

c----
c  get first function; create 1st vel trace
c----

      num = .false.
      iseg = 1

      call  pikrd ( nblk, nseg, iunit, iseg,nsi,itrs,itre,
     1     nsmp,recpik,irecz,itrcz,itz,num,LUDISK,
     2     unit1, unit2)

      do  i = 1, nseg(1)
         if (unit2 .eq. 1) then
            velf(i) = vmin + (itrcz(i)-1) * dvel
         else
            velf(i) = itrcz(i)
         endif
         tf(i) = itz(i)
      enddo

      call vel (tf, velf, nsamp, si, nseg(1), tmp1)
 
c----
c  get second function; create 2nd vel trace
c----

      num = .false.
      iseg = 2

      call  pikrd ( nblk, nseg, iunit, iseg,nsi,itrs,itre,
     1     nsmp,recpik,irecz,itrcz,itz,num,LUDISK,
     2     unit1, unit2)

      do  i = 1, nseg(2)
         if (unit2 .eq. 1) then
            velf(i) = vmin + (itrcz(i)-1) * dvel
         else
            velf(i) = itrcz(i)
         endif
         tf(i) = itz(i)
      enddo

      call vel (tf, velf, nsamp, si, nseg(2), tmp2)
 
      if (tmp1(1) .gt. tmp2(1)) then
         rev = .true.
         call vmov (tmp1, 1, velhi, 1, nsamp)
         call vmov (tmp2, 1, vello, 1, nsamp)
      elseif (tmp1(1) .lt. tmp2(1)) then
         rev = .false.
         call vmov (tmp1, 1, vello, 1, nsamp)
         call vmov (tmp2, 1, velhi, 1, nsamp)
      endif

      do  i = 2, nsamp
         if (velhi(i) .le. vello(i)) then
            write(LERR,*)'Fairway velocity functions cross'
            write(LERR,*)'Ignoring fairway'
            call vclr  (vello, 1, nsamp)
            call vfill (999999., velhi, 1, nsamp)
            go to 111
         endif
      enddo

      if (verbos) then
         write(LERR,*)'vello'
         write(LERR,*)(vello(ii),ii=1,nsamp)
         write(LERR,*)'velhi'
         write(LERR,*)(velhi(ii),ii=1,nsamp)
         write(LERR,*)' '
      endif

c----
c   velocity fairway all read in (or ignored)
c----

 111  continue
c
c--------------------------------------------------------
c  adjust line header, historical l.h., & output header

      ivmin = vmin
      ivmax = vmax
      dvel  = (vmax - vmin)/float(nvel-1)

      call savew( itr, 'MnLnIn',  0   , LINHED)
      call savew( itr, 'ILClIn',  vmin, LINHED)
      call savew( itr, 'CLClIn',  dvel, LINHED)
      call savew( itr, 'MinVel', ivmin, LINHED)
      call savew( itr, 'MaxVel', ivmax, LINHED)

      call savew( itr, 'NumTrc', nvel , LINHED)
      call savew( itr, 'NumRec', nrecc, LINHED)
      call savew( itr, 'NumSmp', istop, LINHED)
      obytes = SZTRHD + SZSMPD * istop
      call savhlh ( itr, lbytes, lbyout )
      call wrtape(luout, itr, lbyout )

c---------------------------------------
c  skip to start record
      call recskp(1,irs-1,luin,ntrc,itr)

C**********************************************************************C
C
C      READ TRACES, COMPUTE SPECTRA, WRITE RESULTS
C
C**********************************************************************C
      ist=0
      DO 5001 JJ = irs, ire

c-----------------------------------------------
c  skip to start trace of current record

         call trcskp(jj,1,ns-1,luin,ntrc,itr)

         ist=ist+1
         k=0

c----------------------------
c  read selected data

         valmx = 0.
         valmy = 0.
         idtot = 0
         iltot = 0

         DO 5002 KK = ns, ne

            NBYTES = 0
            CALL RTAPE  ( LUIN, ITR, NBYTES )
            if(nbytes .eq. 0) then
               write(LERR,*)'End of file on input:'
               write(LERR,*)'  rec= ',jj,'  trace= ',kk
               go to 999
            endif
            call vmov (itr(ITHWP1),1,tri,1,nsamp)
            call saver2(itr,ifmt_StaCor,l_StaCor, ln_StaCor,
     1           istatic , TRACEHEADER)
            call saver2(itr,ifmt_RecNum,l_RecNum, ln_RecNum,
     1           irec , TRACEHEADER)

            IF (istatic .ne. 30000) THEN
               if (kk .eq. ns)
     1              call vmov (itr, 1, str, 1, ITRWRD)
               call saver2(itr,ifmt_DstSgn,l_DstSgn, ln_DstSgn,
     1              idis , TRACEHEADER)
               call saver2(itr,ifmt_DphInd,l_DphInd, ln_DphInd,
     1              idi  , TRACEHEADER)
               call saver2(itr,ifmt_LinInd,l_LinInd, ln_LinInd,
     1              ili  , TRACEHEADER)
               call saver2(itr,ifmt_SrPtXC,l_SrPtXC, ln_SrPtXC,
     1              ivalsx  , TRACEHEADER)
               call saver2(itr,ifmt_SrPtYC,l_SrPtYC, ln_SrPtYC,
     1              ivalsy  , TRACEHEADER)
               call saver2(itr,ifmt_RcPtXC,l_RcPtXC, ln_RcPtXC,
     1              ivalrx  , TRACEHEADER)
               call saver2(itr,ifmt_RcPtYC,l_RcPtYC, ln_RcPtYC,
     1              ivalry  , TRACEHEADER)
               idtot = idtot + idi
               iltot = iltot + ili
               valmx = valmx + .5 * (float (ivalsx+ivalrx)) + 0.5
               valmy = valmy + .5 * (float (ivalsy+ivalry)) + 0.5
               
               k=k+1
               dis(k) = iabs( idis ) * dmul
               istrc  = (k-1) * nsamp
               do 5004 LL= 1, nsamp
                  traces(istrc + LL) = tri(LL)
 5004          continue
            ENDIF

 5002    CONTINUE

         if (k .gt. 0) then
            idtot = idtot / k
            iltot = iltot / k
            valmx = valmx / float(k)
            valmy = valmy / float(k)
         endif

c----------------------------------
c  do spectra

         call tvel(traces,vels,str,tri,dt,nvel,k,dis,thresh,
     1        vmin,vmax,luout,obytes,slowness,irec,logsc,
     2        ntap,jj,ns,ne,nsi,nsamp,istart,istop,weight,
     3        start,stop,igt,linear,abslut,irs,ire,ist,idi,
     4        ifmt_TrcNum,l_TrcNum,ln_TrcNum,vello,velhi,
     5        ifmt_RecNum,l_RecNum,ln_RecNum,
     6        ifmt_DstSgn,l_DstSgn,ln_DstSgn,
     7        ifmt_DphInd,l_DphInd,ln_DphInd,
     8        ifmt_LinInd,l_LinInd,ln_LinInd,
     9        ifmt_StaCor,l_StaCor,ln_StaCor,
     1        ifmt_CDPBCX,l_CDPBCX,ln_CDPBCX,
     2        ifmt_CDPBCY,l_CDPBCY,ln_CDPBCY,
     3        snum,sden,cn,ITRWRD,ITHWP1,SZLNHD,
     4        idtot,iltot,valmx,valmy,non_zero_samples,verbos)

         if(verbos) write(LER,*)'Completed Record= ',JJ

 5001 CONTINUE

 999  continue

      call lbclos( luin)
      call lbclos(luout)

      call exit(0)
      end

C**********************************************************************C
      subroutine tvel ( traces,vels,itr,tri,dt,nvel,k,dis,thresh,
     1     vmin,vmax,luout,obytes,slowness,irec,logsc,
     2     ntap,nrec,ktrc1,ktrc2,nsi,nsamp,istart,istop,
     3     wt,start,stop,igt,linear,abslut,irs,ire,ist,idi,
     4     ifmt_TrcNum,l_TrcNum,ln_TrcNum,vello,velhi,
     5     ifmt_RecNum,l_RecNum,ln_RecNum,
     6     ifmt_DstSgn,l_DstSgn,ln_DstSgn,
     7     ifmt_DphInd,l_DphInd,ln_DphInd,
     8     ifmt_LinInd,l_LinInd,ln_LinInd,
     9     ifmt_StaCor,l_StaCor,ln_StaCor,
     1     ifmt_CDPBCX,l_CDPBCX,ln_CDPBCX,
     2     ifmt_CDPBCY,l_CDPBCY,ln_CDPBCY,
     3     snum,sden,cn,ITRWRD,ITHWP1,SZLNHD,
     4     idtot,iltot,valmx,valmy, non_zero_samples, verbos )
C**********************************************************************C
c      SUBROUTINE ARGUMENTS
c
c      traces    R*4      Two dimensional array of spectra of trace
c                         traces(LL,LR) for trace LR and time LL
c      obytes    I*4      output number of bytes
c      itr       I*2      output data vector equivalenced to...
c...   tri       R*4      ...output real data vector
c      luout     I*4      output data unit number
c      istart    I*4      start time in samples
c      start     R*4      start time in ms
c      istop     I*4      stop time in samples
c      stop      R*4      stop time in ms
c      dt        R*4      Sample interval in MILLISECONDS
c      nvel      I*4      Number of velocity inputs
c      k         I*4      Number of traces
c      dis       R*4      Array of source-receiver distances
c      vmin      R*4      Minimum velocity  (ft/sec)for plot
c      vmax      R*4      Maximum velocity  (ft/sec)for plot
c      ntap      C*255    File name
c      nrec      I*4      Record number
c      ktrc1     I*4      Starting trace number
c      ktrc2     I*4      Ending trace number
c      nsi       I*4      Sample interval in MILLISECONDS
c      NSAMP     I*4      Number of samples in original time series
c      IGT       I*4      integer for time gate centered at zero-offset time
c                         semblance varies between 0 and 1
c      abslut    L        contour absolute semblance rather than relative
C**********************************************************************C
c

#include <f77/iounit.h>
#include <f77/sisdef.h>

c declare variables passed from calling routine

      integer nvel, k, luout, obytes, irec, nrec, ktrc1, ktrc2
      integer nsi, nsamp, istart, istop, igt, irs, ire, ist, idi 
      integer ifmt_TrcNum,l_TrcNum,ln_TrcNum
      integer ifmt_RecNum,l_RecNum,ln_RecNum
      integer ifmt_DstSgn,l_DstSgn,ln_DstSgn
      integer ifmt_DphInd,l_DphInd,ln_DphInd
      integer ifmt_LinInd,l_LinInd,ln_LinInd
      integer ifmt_StaCor,l_StaCor,ln_StaCor
      integer ifmt_CDPBCX,l_CDPBCX,ln_CDPBCX
      integer ifmt_CDPBCY,l_CDPBCY,ln_CDPBCY
      integer ITRWRD, ITHWP1, SZLNHD, idtot, iltot
      integer itr(SZLNHD)
      integer non_zero_samples (SZLNHD)

      real vmin, vmax, dt, stop, start, valmx, valmy, thresh
      real traces(nsamp,k), vels(nsamp,nvel), tri(nsamp)
      real dis(k), wt(k), vello(nsamp), velhi(nsamp)
      real snum(SZLNHD), sden(SZLNHD), cn(SZLNHD)

      character ntap*(*)

      logical abslut, linear, slowness, logsc, verbos

c declare local variables

      integer   IO, IR, LL, LR, IGTP, IGTM, IGL, IGH, IG
      integer   llmax, I, J, ili, ivalmx, ivalmy, it

      real cnnum, cnden, xtrc, amp, vel, dt1, r, tt, wti
      real recmax, v1, v2, dvel, smin2, smax2, ds2, s2, fmax, sem

c initialize variables

      xtrc = k
      dt1 = dt

C**********************************************************************C
c      perform scan of trajectories to determine vrms
C**********************************************************************C
      recmax = 0.0

      if (logsc) then
         v1 = log (vmin)
         v2 = log (vmax)
         dvel = (v2 - v1)/float(nvel-1)
      else
         v1 = vmin
         v2 = vmax
         dvel = (vmax-vmin)/(NVEL-1)
      endif

      if (nrec .eq. irs) then

c only report this for the first spectra calculated

         write(LERR,*)'Velocity increment = ',dvel
         if (verbos) write(LER ,*)'Velocity increment = ',dvel
      endif

      if(slowness) then
         if (logsc) then
            smin2  = log((1./vmax)**2)
            smax2  = log((1./vmin)**2)
            ds2    = (smax2-smin2)/(nvel-1)
         else
            if (linear) then
               smin2 =  1./vmax
               smax2 =  1./vmin
            else
               smin2 = (1./vmax)**2
               smax2 = (1./vmin)**2
            endif
            ds2   = (smax2-smin2)/(nvel-1)
         endif
      endif

C**********************************************************************C
c       for each velocity,
C**********************************************************************C

      DO 100 IR = 1,NVEL
         
         do 45 LL=1,ISTOP
            cn(LL) = 0.
 45      continue

         if(slowness) then
            if (logsc) then
               s2  = exp(smin2+(IR-1)*ds2)
               vel = sqrt(1./s2)
            else
               s2  = smin2+(IR-1)*ds2
               if (linear) then
                  vel = 1./s2
               else
                  vel = sqrt(1./s2)
               endif
            endif
         else
            if (logsc) then
               vel = exp (v1 + float(IR-1)*dvel)
            else
               vel = v1+(IR-1)*dvel
            endif
         endif
            
         DO LL=ISTART,ISTOP
            snum(LL) = 0.
            sden(LL) = 0.
            non_zero_samples(LL) = 0
         ENDDO

C**********************************************************************C

         DO 102 LR = 1,k

            if (linear) then
               if (vel .ne. 0.) then
                  r = (dis(LR)/(vel*dt1))
               else
                  r = 1.e06
               endif
            else
               r = (dis(LR)/(vel*dt1))**2
            endif

            DO 102 LL = ISTART,ISTOP

               IF (vel .ge. vello(LL) .AND. vel .le. velhi(LL)) THEN
                     
                  if (linear) then
                     tt =      (LL-1)   + r  + 1.
                  else
                     tt = sqrt((LL-1)**2+ r) + 1.
                  endif
                  
                  it = tt

c load the trace amplitude for this position on the velocity scan

                  if(it.lt.1)then

c samples above the data are not allowed

                     amp = 0.0
                  elseif(it.ge.nsamp)then

c samples from below the data are also not allowed

                     amp = 0.0
                  else
                     amp = traces(it,LR)

                     if ( abs(amp) .gt. 1.e-32 ) 
     :                    non_zero_samples(LL) = 
     :                    non_zero_samples(LL) + 1
                  endif

c load up numerator and denominator terms for the semblance calculation

                  snum(LL)=snum(LL) + amp
                  sden(LL)=sden(LL) + amp*amp
               ELSE
                  snum(LL) = 0.
               ENDIF
                  
 102     CONTINUE

C**********************************************************************C
c      square numerator term
C**********************************************************************C

         DO 103 LL=ISTART,ISTOP
            snum(LL)=snum(LL)**2
 103     continue

C**********************************************************************C
c      decide for a time gate
C**********************************************************************C

         IGTP = ISTART + IGT
         IGTM = ISTOP  - IGT
         IO = IGTM - IGTP + 1

C**********************************************************************C
c      begin to compute semblance for a given velocity
C**********************************************************************C

         DO 200 LL=IGTP,IGTM

c fill out semblance trace for window requested.  have to start
c igt below start and above end to allow for smoothing

            cnnum = 0.
            cnden = 0.

            IGL = LL-IGT
            if (IGL.le.0) IGL = 1
            IGH = LL+IGT
            if (IGH.gt.ISTOP) IGH = ISTOP

            ig = 0

            DO 305 I=IGL,IGH
               ig = ig + 1
               wti = wt (ig)

c do cosine weighting of numerator 

               cnnum = cnnum + snum(I) * wti
               cnden = cnden + sden(I)
 305        continue

C**********************************************************************C
c
c   put correlation z-values into big matrix

            if(cnden.ne.0.0)then
               cn(LL)= cnnum/(xtrc*cnden)
               cn(LL) = cn(LL) * ( 2.0 - (float(non_zero_samples(LL))
     :              / xtrc))
            endif

 200     continue
C**********************************************************************C
c
c   if absolute is true do not normalize correlation in time

         call vclr ( tri,1,istop )

         if(.not.abslut) then
            fmax=0.
            
            do 210 LL=ISTART,ISTOP
               if(cn(LL) .ge. fmax) then 
                  fmax = cn(LL)
                  llmax = LL
               endif
 210        continue
            
            if(fmax .ne. 0.) then
               
               do 220 LL=ISTART,ISTOP
                  sem = cn(LL)/fmax
                  if (sem .lt. thresh) sem = 0.0
                  tri(LL) = sem
 220           continue

            endif
         
         else
            
            do 230 LL=ISTART,ISTOP
               sem = cn(LL)
               if (sem .lt. thresh) sem = 0.0
               tri(LL) = sem
 230        continue
            
         endif

         call vmov (tri, 1, vels(1,IR), 1, nsamp)

 100  continue

C**********************************************************************C
      
      do  1000  J = 1, nvel

c write output semblance spectra with appropriate header information
c loaded.

         call vmov (vels(1,J), 1, tri, 1, nsamp)

         if (logsc) then
            vel= exp (v1 + float(J-1)*dvel)
         else
            vel= v1+(J-1)*dvel
         endif
         
         ivel = vel

         call savew2(itr,ifmt_RecNum,l_RecNum, ln_RecNum,
     1        irec , TRACEHEADER)
         call savew2(itr,ifmt_TrcNum,l_TrcNum, ln_TrcNum,
     1        J   , TRACEHEADER)
         call savew2(itr,ifmt_DstSgn,l_DstSgn, ln_DstSgn,
     1        ivel , TRACEHEADER)

         idi = idtot
         ili = iltot

         call savew2(itr,ifmt_DphInd,l_DphInd, ln_DphInd,
     1        idi  , TRACEHEADER)
         call savew2(itr,ifmt_LinInd,l_LinInd, ln_LinInd,
     1        ili  , TRACEHEADER)
         call savew2(itr,ifmt_StaCor,l_StaCor, ln_StaCor,
     1        0   , TRACEHEADER)

         ivalmx = valmx
         ivalmy = valmy

         call savew2(itr,ifmt_CDPBCX,l_CDPBCX, ln_CDPBCX,
     1        ivalmx, TRACEHEADER)
         call savew2(itr,ifmt_CDPBCY,l_CDPBCY, ln_CDPBCY,
     1        ivalmy, TRACEHEADER)

         call vmov  (tri, 1, itr(ITHWP1), 1, nsamp)
         call wrtape(luout, itr, obytes)

 1000 continue

      return
      end

C**********************************************************************C
      function rinter(y1,y2,dx)
C**********************************************************************C
c      interpolate between y1 and y2 for 0 < dx < 1
C**********************************************************************C
      rinter = (1.0 - dx)*y1 + dx*y2
      return
      end

c-----------------------------------------
c  online help section
c-----------------------------------------
       subroutine  help
#include <f77/iounit.h>

         write(LER,*)
     :'***************************************************************'
      write(LER,*)' '
      write(LER,*)
     :'Execute program by typing velspec and the following arguments'
      write(LER,*)
     :' -N[ntap]    (default stdin)     : Input data file name'
      write(LER,*)
     :' -O[otap]    (default stdout)    : Output data file'
      write(LER,*)
     :' -F[fpik]    (default ignore)    : fairway picks from xsd'
      write(LER,*) ' '
      write(LER,*)
     :' -tmin[start] (deflt=0)          : start time of spectra (ms)'
      write(LER,*)
     :' -tmax[stop] (deflt = full trace):stop time of spectra (ms)'
      write(LER,*)
     :' -rs[irs],     (default=1)  : beginning record number'
      write(LER,*)
     :' -re[ire],   (default=last) : ending record number'
      write(LER,*)
     :' -ns[ns],      (default=1)  : starting trace number'
      write(LER,*)
     :' -ne[ne]     (default=last) : ending trace number'
      write(LER,*) ' '
      write(LER,*)
     :' -nvel[nvel] (default=101)  :         number of velocities'
      write(LER,*)
     :' -igt[igt]    (default=8)   : samples in time gate(= 2*igt*dt)'
      write(LER,*)
     :' -dmul[dmul]    (default=1) : distance multiplier'
      write(LER,*)
     :' -tmul[tmul]   (default=1)  : sampling interval multiplier'
      write(LER,*)
     :' -vmin[vmin]  (deflt=1000)  : minimum velocity (ft/sec) for plot'
      write(LER,*)
     :' -vmax[vmax]  (deflt=10000) : maximum velocity(ft/sec)for plot'
      write(LER,*)
     :' -thld[thresh]  (deflt=0.0) : zero semblances < this value'
      write(LER,*) ' '
      write(LER,*)
     :' -V      (default = false)  : verbose output'
      write(LER,*)
     :' -L      ( default false )  : do linear velocity analysis'
      write(LER,*)
     :' -A      ( default false )  : plot scaled semblance contours'
      write(LER,*)
     :' -S      ( default false )  : horizontal axis is slowness**2'
      write(LER,*)
     :' -W      ( default cosine ) : no cosine weighting'
      write(LER,*)
     :' -log    ( default false )  : velocity (or slowness scale) is'
      write(LER,*)
     :'                              logarithmic'
      write(LER,*) ' '
      write(LER,*)
     :'USAGE: velspec -N[ntap] -re[re] -rs[rs] -ns[ns] -ne[ne]'
      write(LER,*)
     :'               -vmin[vmin] -vmax[vmax] -F[fpik]'
      write(LER,*)
     :'               -nvel[nvel] -tmin[start] -tmax[stop] -igt[igt] '
      write(LER,*)
     :'               -tmul[] -dmul[] -thld[]  [-W -log -L -V -A -S]'
         write(LER,*)
     :'***************************************************************'

      return
      end

c-----
c     get command arguments
c
c     ntap  - C*100  input file name
c     otap  - C*100  output file name
c      ns   - I      start trace
c      ne   - I      stop end trace
c     irs   - I      start record
c     ire   - I      stop end record
c   start   - R      start time of analysis
c    stop   - R      stop time
c    dmul   - R      distance multiplier
c    tmul   - R      sample interval multiplier
c    nvel   - I      number velocities
c    vmin   - R      min velocity
c    vmax   - R      max velocity
c    abslut - L      use absolute semblance
c    verbos - L      verbose output or not
c-----
      subroutine cmdln (ntap,otap,start,stop,dmul,tmul,igt,
     &               ns,ne,irs,ire,nvel,vmin,vmax,abslut,verbos,
     &               linear,slowness,thresh,fpik,logsc,nowt)
#include <f77/iounit.h>
#include <f77/lhdrsz.h>
      character  ntap*(*), otap*(*), fpik*(*)
      integer    argis,ns,ne,irs,ire,nvel,igt,start,stop
      real       dmul,tmul,vmin,vmax
      logical    abslut, verbos, linear,slowness,logsc,nowt

         call argstr('-N', ntap, ' ', ' ' )
         call argstr('-O', otap, ' ', ' ' )
         call argstr('-F', fpik, ' ', ' ' )
         call argr4 ('-tmin', START, 0., 0.)
         call argr4 ('-tmax', STOP , 0., 0.)
         call argr4('-dmul',dmul,1.0,1.0)
         call argr4('-tmul',tmul,1.0,1.0)
         call argi4('-igt',IGT,8,8)
         call argi4 ('-rs',irs,1,1)
         call argi4 ('-re',ire,0,0)
         call argi4 ('-ns',ns,1,1)
         call argi4 ('-ne',ne,0,0)
         call argi4 ('-nvel',nvel,101,101)
         call argr4('-vmin',vmin,1000.,1000.)
         call argr4('-vmax',vmax,10000.0,10000.0)
         call argr4('-thld',thresh,0.0,0.0)
         nowt     = ( argis ('-W') .gt. 0 )
         linear   = ( argis ('-L') .gt. 0 )
         logsc    = ( argis ('-log') .gt. 0 )
         verbos   = ( argis ('-V') .gt. 0 )
         abslut   = ( argis ('-A') .eq. 0 )
         slowness = ( argis ('-S') .gt. 0 )

         if (fpik(1:1) .ne. ' ') then
            open (unit=LUDISK,file=fpik,status='old',iostat=ierr)
 
            if (ierr .ne. 0) then
               write(LERR,*)'Could not open velpik file ',fpik
               write(LERR,*)'Check existence'
               fpik = ' '
            endif
         endif

      ierror=0
      if(nvel.gt.SZSPRD) then
         write(lerr,*) 'command line error!'
         write(LERR,*)'Number velocities entered after -nvel'
     1                //' arguement too high!'
         write(LERR,*)'Must be less than ',SZSPRD
         ierror=ierror+1
      endif
      if(vmin .eq. 0. .and. slowness) then
         write(lerr,*) 'command line error!'
         write(lerr,*) 'cannot input vmin = 0. after -vmin argument '
     1                 //'when -S option is used'
         write(lerr,*) 'please enter a vmin such that slowness '
     1                 //' smax=1./vmin is well defined'
         ierror=ierror+1
      endif
      if(ierror .ne. 0) then
         write(lerr,*) 'program aborted due to ',ierror,
     1                 ' command line errors!'
         call exit(666)
      endif

      return
      end
