/***********************************************************************
 *                copyright 2001, Amoco Production Company             *
 *                            All Rights Reserved                      *
 *                    an affiliate of BP America Inc.                  *
 ***********************************************************************/
/*
 *	modified test version with gzip compression
 * 	(part of xsd.new) J. Ehlers 4/2/98
 */
/* #include <malloc.h> */
#include <signal.h>
#include <fcntl.h>
#include <stdio.h>
#include <string.h>
#include <unistd.h>	/* pipe */
#include <stdlib.h>	/* close */

#include <localsys.h> /* which system */
#include <ut_defs.h> /* byte swap stuff */

typedef char	Boolean;

extern Boolean	verbose, server_debug;
extern char	*ServerName;

void	amoco_start_server(char*, char*, char*, char*);
int	amoco_check_server(char*);
void	amoco_color_range(int, int);
void	amoco_detect_format(void);
void	amoco_specify_format(char*);
int	amoco_read_line_header(char**, char*);
int	amoco_read_data(unsigned char*, unsigned char*, char*, 
			int, int, int, int, int, int, int, int, int,
			int, int, float, float, int, Boolean, Boolean);
int	amoco_offset(void);
float	amoco_scalar(void);
int	amoco_ls_size(char*);
int	amoco_ls_files(char*, int);
void	amoco_end_server(void);

static int	amoco_reader(int, unsigned char*);

static FILE	*out_pipe,*in_pipe;
static int	sk_client[2],sk_server[2];
static float	scalar;
static int	offset,seq_number=0;
static char	buf[1024];


/*
  Start the server program.
*/
void amoco_start_server(char *hostname, char *login, 
			char *command, char *machine)
{
  char *hostalias=NULL;
				
  /* Ignores the pipe errors. */
  signal(SIGPIPE,SIG_IGN);

  /* Opens two pipes. */
  if ((pipe(sk_server)<0)||(pipe(sk_client)<0)) {
    fprintf(stderr,"Error openning connection.\n");
    return;
  }

  if (fork()==0) {
    if (sk_server[0]!=0) {
      close(0);
      if (dup(sk_server[0])!=0) {
	fprintf(stderr,"Error openning connection.\n");
	exit(-1);
      }
      close(sk_server[0]);
    }
    if (sk_client[1]!=1) {
      close(1);
      if (dup(sk_client[1])!=1) {
	fprintf(stderr,"Error openning connection.\n");
	exit(-1);
      }
      close(sk_client[1]);
    }
    close(sk_server[1]);
    close(sk_client[0]);

    /* DWN 7/13/98 lookup alias list for hosts */

    if ((hostalias=getenv("XSD_SERVER_ALIAS_TABLE")) == NULL) {
      fprintf(stderr, "Can't find XSD_SERVER_ALIAS_TABLE, no server host alias used.\n");
    } else {
      FILE *fd=NULL;

      fd=fopen(hostalias,"r");
      if (fd==NULL) {
        fprintf(stderr,"Can't open %s, no server host alias used.\n",hostalias);
      } else {
        int n=2;
        char oldhost[256],newhost[256];

        while (n>0) {
          n=fscanf(fd,"%s %s",oldhost,newhost);
          if (n==2) {
            if (strcmp(oldhost,hostname)==0) {
              fprintf(stderr,"xsd: changed server host %s to %s\n",oldhost,newhost);
              strncpy(hostname,newhost,20);
              hostname[19]='\0';
            }
          }
        }
        fclose(fd);
      }
    }

    if (hostname[0]=='=') {
      if (verbose) fprintf(stderr,"Starting local data server %s.\n",command);
      if (server_debug) {
	execlp(command,command,machine,"xsd_debug=",(char *)0);
      } else {
	execlp(command,command,machine,(char *)0);
      }
    } else {
      if (verbose) fprintf(stderr,"Starting remote data server on %s.\n",hostname);

      /* DWN 6/30/98 change ucb to bin */
      /* on SunOS /usr/ucb/rsh */
      /* on Solaris /usr/ucb/rsh is a symbolic link to /usr/bin/rsh */
      /* on Linux it is in /usr/bin/rsh */
      /*
	#if defined(__sgi)
	execl("/usr/bsd/rsh",hostname,"-l",login,command,machine,(char *)0);
	#else
	execl("/usr/bin/rsh",hostname,"-l",login,command,machine,(char *)0);
	#endif
	*/

      /* DWN JWE 1/20/98 we expect user to find rsh in their path */
      if (server_debug) {
	execlp("rsh",hostname,"-l",login,command,machine,"xsd_debug=",(char *)0);
      } else {
	execlp("rsh",hostname,"-l",login,command,machine,(char *)0);
      }
    }
  }

  close(sk_server[0]);
  close(sk_client[1]);

  out_pipe=fdopen(sk_server[1],"w");
  if (out_pipe==NULL) {
    fprintf(stderr,"Error openning write connection.\n");
    exit(-1);
  }
  in_pipe=fdopen(sk_client[0],"r");
  if (in_pipe==NULL) {
    fprintf(stderr,"Error openning write connection.\n");
    exit(-1);
  } 
}


/*
  Send a command to the server to check that it still running.
*/
int amoco_check_server(char *buf)	/* note: buf size assumed to be 1024 */
{
  const int	BufSize = 1024;
  int		i, n, len, rc;

  rc = 0;
  seq_number++;

  fprintf(out_pipe, "%d Check\n", seq_number);
  fflush(out_pipe);

  n = read(sk_client[0], buf, 3);

  if (verbose) {
    fprintf(stderr, "amoco_check_server: got %d bytes back from server\n",n);
    fprintf(stderr, "amoco_check_server: sequence number %d\n",seq_number);
    fprintf(stderr, "amoco_check_server: return buf=");
    for (i = 0; i < n; i++) fprintf(stderr, "%c", buf[i]);
    fprintf(stderr, "\n");
  }

  if (0 != strncmp(buf, "OK", 2) || n != 3) {
    if (n > 0) {
      len = read(sk_client[0], &buf[n], BufSize-n-1);
      len += n;
      buf[len] = '\0';

      fprintf(stderr, "amoco_check_server: anticipated \"OK\", got \"");
      for(i = 0; i < len; i++) fprintf(stderr, "%c", buf[i]);
      fprintf(stderr, "\"\n");
    } else {
      fprintf(stderr, "amoco_check_server: anticipated \"OK\", got no reply.\n");
      fprintf(stderr, "Server appears to have died.\n");

      sprintf(buf,"");
    }

    rc = -1;
  }

  return rc;
}


/*
  Send a command to the server to set the color range of the raster data.
*/
void amoco_color_range(int start, int end)
{
  seq_number++;
  fprintf(out_pipe,"%d ColorR %d %d\n",seq_number,start,end);
  fflush(out_pipe);
}


void amoco_detect_format(void)
{
  seq_number++;
  fprintf(out_pipe, "%d AutoFmt\n", seq_number);
  fflush(out_pipe);
}


void amoco_specify_format(char *format)
{
  seq_number++;
  fprintf(out_pipe, "%d SpecFmt %s\n", seq_number, format);
  fflush(out_pipe);
}


/*
  Send a command to the server for reading the line header of the data.
*/
int amoco_read_line_header(char **header, char *file)
{
	char	type[4];
	int	rc, nbytes,obytes;

	seq_number++;
	fprintf(out_pipe,"%d LineH %s\n",seq_number,file);
	fflush(out_pipe);

	buf[0] = '\0';
	read(sk_client[0],buf,14);

	sscanf(buf,"%3s %d",type,&nbytes);
	if (nbytes==-1)	/* RJM: I don't this this ever happens! */
{ fprintf(stderr, "amoco_read_line_header received nbytes of -1 !!!!!\n");
		return -91;
}

	rc = atoi(type);
	if (0 != rc)	/* only = 0 if type is nonnumeric string */
		if (0 != strstr(ServerName, "dds"))
			return rc;	/* DDS data server; rc is < 0 */
		else
			return -90;

	*header=(char *)malloc(nbytes);

	rc = amoco_reader(nbytes,(unsigned char *)*header);
	if (0 != rc)
		return rc;

#ifdef LINUXSYSTEM
   sisinlh((char *)*header,&nbytes,(char *)*header,&obytes);
   nbytes = obytes;
#endif


	if (strcmp(type,"SIS")==0)
		return 1;
	if (strcmp(type,"VDS")==0)
		return 2;

	return -92;
}


/*
  Send a command to the server for reading the data. 
*/
int amoco_read_data(unsigned char *raster, unsigned char *traceH,char *file, 
		    int srec, int erec, int irec, 
		    int strace, int etrace, int itrace, 
		    int ssamp, int esamp, int isamp, 
		    int type, int s_offset, float s_scalar, 
		    float s_exponent, int sort_mode, Boolean trace_flag,
		    Boolean invert_flag)
{
  int level, n;	/* zip compression level, 0=no compression, 9=slowest */
  int nin, nout;
  int nbytes,error,nbytes_trace;
  unsigned char *data;
#ifdef LINUXSYSTEM
  int itr;
  int thlen;
  thlen=256;
#endif

  level = 1;	/* use zip level of 1, appears to be best overall */

  seq_number++;

  if (sort_mode!=0) /* Sorted data request. */
    fprintf(out_pipe,
       "%d Data %s %d %d %d %d %d %d %d %d %d %d %d %e %e %d %d %d\n",
       seq_number,file,srec,erec,irec,strace,etrace,itrace,ssamp,esamp,
       isamp,type,s_offset,s_scalar,s_exponent, 
       trace_flag?1:0,invert_flag?1:0,level);
  else /* Unsorted data request. */
    fprintf(out_pipe,
       "%d DataNS %s %d %d %d %d %d %d %d %d %d %d %d %e %e %d %d %d\n",
       seq_number,file,srec,erec,irec,strace,etrace,itrace,ssamp,esamp,
       isamp,type,s_offset,s_scalar,s_exponent,
       trace_flag?1:0,invert_flag?1:0,level);

  fflush(out_pipe);

  buf[0] = '\0';
  error=read(sk_client[0],buf,80);
  if (error > 0) buf[error] = '\0';
  if (error != 80) {
    fprintf(stderr, "XSD ERROR: only read %d bytes of 80 from ddsServer3!\n", error);
  }

  nbytes = 0;
  offset = 0;
  scalar = 0.0;
  nbytes_trace = 0;
  level = 0;
  n = sscanf(buf,"%d %d %e %d %d",&nbytes,&offset,&scalar,&nbytes_trace, &level);
  if (n != 5) {
    fprintf(stderr, "XSD ERROR: only read %d values from ddsServer3 buf='%s'\n",
	    n, buf);
  }
  
  if (level < 0 || level > 9) level = 0;

  if (verbose) {
    fprintf(stderr,"%d Data? %s %d %d %d %d %d %d %d %d %d %d %d %e %e\n",
	    seq_number,file,srec,erec,irec,strace,etrace,itrace,ssamp,
	    esamp,isamp,type,s_offset,s_scalar,s_exponent);
    fprintf(stderr,"Scaling %e %d\n",scalar,offset);
    fprintf(stderr,"%d bytes of data being transfered\n",nbytes);
    fprintf(stderr,"Compression level %d\n",level);
  }

  if (nbytes < 0) return nbytes;

  if (level > 0 && nbytes > 0) {
    data = (unsigned char*)malloc(nbytes);
    error=amoco_reader(nbytes,data);
    if (0 == error) 
    {
      nin = nbytes;
      unzip(&nbytes, &data);
      nout = nbytes;
      memcpy(raster, data, nbytes);
    }
    free((char*)data);
  } else {
    data = raster;
    error=amoco_reader(nbytes,data);
  }
  if (0 != error) return error;

  if (verbose)
    fprintf(stderr,"%d bytes of data being transfered\n",nbytes_trace);

  if (nbytes_trace==-1)
{ (void)fprintf(stderr, "amoco_read_data: THE IMPOSSIBLE HAPPENED; nbytes_trace==-1!!!.\n");
    return -93;
}


  if (level > 0 && nbytes_trace > 0) {
    data = (unsigned char*)malloc(nbytes_trace);
    error=amoco_reader(nbytes_trace,data);
    if (0 == error) 
    {
      nin += nbytes_trace;   
      unzip(&nbytes_trace, &data);
#ifdef LINUXSYSTEM
    for (itr=0; itr<nbytes_trace; itr+=thlen) {
      sisint3(data+itr,&thlen,data+itr,&thlen);
    }
#endif
      nout += nbytes_trace;
      memcpy(traceH, data, nbytes_trace);
    }
    free((char*)data);
  } else {
    data = traceH;
    error=amoco_reader(nbytes_trace, data);
#ifdef LINUXSYSTEM
    for (itr=0; itr<nbytes_trace; itr+=thlen) {
      sisint3(data+itr,&thlen,data+itr,&thlen);
    }
#endif
  }
  
  if (verbose) {
    if (level > 0) fprintf(stderr, "%d%% compression\n", 
    		(100*(nout-nin))/nout);
    else fprintf(stderr, "No compression done\n");
  }
  
  return error;
}


int amoco_offset(void)
{
  return(offset);
}


float amoco_scalar(void)
{
  return(scalar);
}
	  

/*
  Find the length needed to store the file listing.
*/
int amoco_ls_size(char *file)
{
  int nbytes;

  seq_number++;
  fprintf(out_pipe,"%d Files %s\n",seq_number,file);
  fflush(out_pipe);
  read(sk_client[0],buf,10);
  buf[9]='\0';
  sscanf(buf,"%d",&nbytes);
  if (verbose)
    fprintf(stderr,"%d bytes of data being transfered\n",nbytes);
  if (nbytes==-1)
    return(-1);

  return(nbytes);
}


/*
  Retrieve the file listing.
*/
int amoco_ls_files(char *string, int nbytes)
{
  int error;
  char *data;

  data = string;
  error=amoco_reader(nbytes,(unsigned char *)data);

  return error;
}


/* 
  Send a command to the server to stop the server program. 
*/
void amoco_end_server(void)
{
  seq_number++;
  fprintf(out_pipe,"%d End\n",seq_number);
  fflush(out_pipe);
}


/*
  Read from the pipe and prints a hash mark ever 10,000 bytes.
*/
int amoco_reader(int nbytes, unsigned char *data)
{
  int n,l,count,hash_count;

  if (verbose)
    fprintf(stderr,"\n");

  hash_count=0;
  count=0;

  for(n=nbytes; n > 0; n -= l) {
    l = read(sk_client[0], data, n);

    if (l==0)                     /* No data received. */
    {
      if (verbose && count == 0)
	(void)fprintf(stderr, "amoco_reader: no data received\n");

      count++;
    }

    if ((count>1000)||(l<0))     /* After 1000 tries assume error. */
    {
      if (verbose && count > 1000)
	(void)fprintf(stderr, "amoco_reader: no data received after 1000 tries\n");
      return -94;
    }

    data += l;
    hash_count+=l;

    if (hash_count>10000) {
      hash_count=0;

      if (verbose)
	fprintf(stderr,"#");
    }
  }
  if (verbose)
    fprintf(stderr,"\n");

  return 0;
}
