/***********************************************************************
 *                copyright 2001, Amoco Production Company             *
 *                            All Rights Reserved                      *
 *                    an affiliate of BP America Inc.                  *
 ***********************************************************************/
/*
	Package	:	sisio
	Module	:	_flushit.c
	Author	:	Dean Kopesky
	Company	:	Minnesota Supercomputer Center, Inc.


	_Flushit flushes the buffer associated with a unit with large
	buffering enabled.  It is an error to flush the buffer if large
	buffering is not enabled.

	error = _flushit( unit );

	error	return	int	Error code, 0 if OK, < 0 if error.
	unit	input	int	Unit to flush the buffer for.


	$Header: /m/s1/dmk/xlate/sisio/RCS/_flushit.c,v 4.1 90/07/13 08:56:55 dmk Exp $

	$Log:	_flushit.c,v $
	Revision 4.1  90/07/13  08:56:55  dmk
	MODSET: 4.1
	Added check for format L.
	
	Revision 1.2  88/06/24  16:21:52  dmk
	MODSET: 1.2
	It is no longer an error to flush the buffer if no I/O has occurred.
	
	Revision 1.1  88/06/24  12:48:10  dmk
	Initial revision
	
*/

#ifdef DEBUG
#include <stdio.h>
#endif

#include	<stdlib.h>
#include	<sisio.h>
/* 8/27/01 - had to add these for switch to off_t variable usage in lseek */
#include	<sys/types.h>
#include	<unistd.h>


int _flushit( unit )

	int	unit;

{
	off_t	byte;
	int	nwritten;
	off_t	pos;

	/*
		Check for large buffering.
	*/

	if ( ! _sisfit[ unit ].largebuf ) return( -E_INTERNAL );

	/*
		If we have been reading from the file, flush as follows.
	*/

	if ( _sisfit[ unit ].lgoper == L_READ )
	{
		/*
			If pointers in synch, no need to seek.
		*/

		if ( _sisfit[ unit ].bfrbeg >= _sisfit[ unit ].bfrend )
		{
			_sisfit[ unit ].bfrbeg = 0;
			_sisfit[ unit ].bfrend = 0;

			return( 0 );
		}

		/*
			Otherwise, fall through to here and figure out where
			we should be in the file.
		*/

		if ( _sisfit[ unit ].pointer == 0 )
		{
			byte = 0;
		}
		else if ( _sisfit[ unit ].pointer == 1 )
		{
			if ( _sisfit[ unit ].hdrlen == 0 ||
					_sisfit[ unit ].cwlen == 0 )
				return( -E_INTERNAL );

			byte = _sisfit[ unit ].hdrlen + _sisfit[ unit ].cwlen;
		}
		else  /* pointer > 1 */
		{
			/*
				Can't handle this case on format L currently.
			*/
			if ( _sisfit[ unit ].format == 'L' )
			{
				_error( "_flushit", unit, E_NOTONL );
				exit( E_SISIO + E_NOTONL );
			}

			if ( _sisfit[ unit ].hdrlen == 0 ||
					_sisfit[ unit ].trclen == 0 ||
					_sisfit[ unit ].cwlen == 0 )
				return( -E_INTERNAL );

		byte = (off_t) ((off_t)  _sisfit[ unit ].hdrlen + 
			 (off_t) _sisfit[ unit ].cwlen + 
			((off_t)( _sisfit[ unit ].pointer - 1 ) * (off_t)
			( _sisfit[ unit ].trclen + _sisfit[ unit ].cwlen )));
		}

		pos = lseek( _sisfit[ unit ].filedes, ( off_t ) byte, 0 );

		if ( pos != byte ) return( -E_LSEEK );

		_sisfit[ unit ].bfrbeg = 0;
		_sisfit[ unit ].bfrend = 0;

		return( 0 );
	}

	/*
		If we have been writing to the file, flush as follows.
	*/

	if ( _sisfit[ unit ].lgoper == L_WRITE )
	{

#ifdef DEBUG
	fprintf(stderr,"_flushit: attempting to write %d bytes to unit %d\n",
		( unsigned ) ( _sisfit[ unit ].bfrend ), unit);
#endif

		nwritten = write( _sisfit[ unit ].filedes,
			_sisfit[ unit ].buffer,
			( unsigned ) _sisfit[ unit ].bfrend );

#ifdef DEBUG
	fprintf(stderr,"_flushit: %d bytes actually written to unit %d\n",
		nwritten, unit);
#endif


		if ( nwritten != _sisfit[ unit ].bfrend ) return( -E_WRITE );

		_sisfit[ unit ].bfrend = 0;

		return( 0 );
	}

	/*
		Otherwise, just return.
	*/

	return( 0 );
}
