C***********************************************************************
C                                                                      *
c                 copyright 2001, Amoco Production Company             *
c                             All Rights Reserved                      *
c                     an affiliate of BP America Inc.                  *
C***********************************************************************
C***********************************************************************
C                                                                      *
C  ROUTINE:       FFT2DI                                               *
C  ROUTINE TYPE:  SUBROUTINE                                           *
C  PURPOSE:                                                            *
C      2D COMPLEX DATA IS FILTERED, TRANSPOSED, AND PUT IN FFT ORDER AN*
C      IS THEN TRANSFORMED FROM THE F-K DOMAIN TO THE T-X DOMAIN TO    *
C      PRODUCE AN NT BY NX REAL OUTPUT MATRIX.                         *
C                                                                      *
C  ENTRY POINTS:                                                       *
C      FFT2DI  (MT,NT,NX,NK,NW,IW1,FILTER,DATAIN,DATAOUT)              *
C  ARGUMENTS:                                                          *
C      MT      INTEGER  ??IOU*      -                                  *
C      NT      INTEGER  ??IOU*      -                                  *
C      NX      INTEGER  ??IOU*      -                                  *
C      NK      INTEGER  ??IOU*      -                                  *
C      NW      INTEGER  ??IOU*      -                                  *
C      IW1     INTEGER  ??IOU*      -                                  *
C      FILTER  REAL     ??IOU*  (*) -                                  *
C      DATAIN  COMPLEX  ??IOU*  (*) -                                  *
C      DATAOU  REAL     ??IOU*  (*) -                                  *
C       +------------------------------------------------------+       *
C       |               DEVELOPMENT INFORMATION                |       *
C       +------------------------------------------------------+       *
C  AUTHOR:   ???                                ORIGIN DATE: 97/02/13  *
C  LANGUAGE: FORTRAN 77                  DATE LAST COMPILED: 97/02/18  *
C       FORTRAN 77                                                     *
C                                                                      *
C  HISTORY:                                                            *
C       ORIGINAL                NOV 92          R.D. COLEMAN, CETech   *
C                                                                      *
C  CALLING FORMAT:                                                     *
C       CALL FFT2DI (MT, NT, NX, NK, NW, IW1, FILTER, DATAIN, DATAOUT) *
C                                                                      *
C  PARAMETERS:                                                         *
C       MT      INTEGER INPUT SCALAR                                   *
C               Row dimension of matrix DATAOUT (MT >= NT+2).          *
C                                                                      *
C       NT      INTEGER INPUT SCALAR                                   *
C               Number of T's (times).                                 *
C                                                                      *
C       NX      INTEGER INPUT SCALAR                                   *
C               Number of X's.                                         *
C                                                                      *
C       NK      INTEGER INPUT SCALAR                                   *
C               Number of K's (wave numbers) (NK >= NX).               *
C                                                                      *
C       NW      INTEGER INPUT SCALAR                                   *
C               Number of omega's (frequencies) (NW <= NT/2).          *
C                                                                      *
C       IW1     INTEGER INPUT SCALAR                                   *
C               Index of first frequency.                              *
C                                                                      *
C       FILTER  REAL INPUT VECTOR OF LENGTH NW                         *
C               Contains the filter coefficients.                      *
C                                                                      *
C       DATAIN  COMPLEX INPUT MATRIX OF DIMENSION MAX( NK*NW, NWORK )  *
C               Source matrix (NK by NW complex elements).  Note:      *
C               DATAIN is also used for work space during processing.  *
C               The amount of work space required, NWORK = (MWORK+1)/2,*
C               is system dependent.  For Cray systems, MWORK =        *
C               MAX( 2*NX*NT+2*NT, 4*NW*NK+2*NK ) real elements.  For  *
C               non-Cray systems, MWORK = MAX( 9*NT/2+18*NX+41,        *
C               6*NK+18*NW+41) real elements.                          *
C                                                                      *
C       DATAOUT REAL OUTPUT MATRIX OF DIMENSION MT BY NK               *
C               Contains the results in the NT by NX principle         *
C               submatix.                                              *
C                                                                      *
C  DESCRIPTION:                                                        *
C       FFT2DI performs the following:                                 *
C          (1) Transposes the data to form the NW by NK complex output *
C              matrix.                                                 *
C          (2) Rearranges the data into FFT order as opposed to natural*
C              order.                                                  *
C          (3) Filters the data; i.e., multiplies every element of the *
C              i-th complex row by real scalar FILTER(i).              *
C          (4) Performs a complex-to-complex inverse FFT of length NK o*
C              the NW complex rows.                                    *
C          (5) Expands the NW rows to NT/2+1 rows such that the origina*
C              NW rows start in row IW1 and fills the other rows with  *
C              zeroes.                                                 *
C          (6) Performs a complex-to-real inverse FFT of length NT on t*
C              first NX columns.                                       *
C                                                                      *
C       The FFT's are performed in place in DATAOUT, thus, it must be  *
C       dimensioned MT by NK.  MT must at least NT+2 to provide space  *
C       for the Nyquist frequency.  In addition to these two extra rows*
C       an additional row should be added to make the row dimension odd*
C       in order to optimize performance on some systems.  Since NT mus*
C       be an even number, MT should be set to MT = NT + 3 + 2*k where *
C       is a non-negative integer.  Since DATAIN is also used for work *
C       space, the input data is destoyed.                             *
C                                                                      *
C  SUBPROGRAMS CALLED:                                                 *
C       FFTFAX, CFTFAX, RFFTMLT, CFFTMLT, VCLR, CVSMUL                 *
C                                                                      *
C  ERROR CONDITIONS:                                                   *
C       None.                                                          *
C                                                                      *
C--------------------------------------------------------------------- *
C                                                                      *
C       +------------------------------------------------------+       *
C       |                 EXTERNAL ENVIRONMENT                 |       *
C       +------------------------------------------------------+       *
C  ROUTINES CALLED:                                                    *
C      VCLR   -                                                        *
C      CVSMUL -                                                        *
C      CFTFAX -                                                        *
C      CFFTML -                                                        *
C      FFTFAX -                                                        *
C      RFFTML -                                                        *
C  INTRINSIC FUNCTIONS CALLED:  NONE                                   *
C  FILES:            NONE                                              *
C  COMMON:           NONE                                              *
C  STOP CODES:       NONE                                              *
C       +------------------------------------------------------+       *
C       |             OTHER DOCUMENTATION DETAILS              |       *
C       +------------------------------------------------------+       *
C  ERROR HANDLING:  ???                                                *
C  GENERAL DESCRIPTION:  ???                                           *
C  REVISED BY:  ???                           REVISION DATE: ?Y/?M/?D  *
C       +------------------------------------------------------+       *
C       |                 ANALYSIS INFORMATION                 |       *
C       +------------------------------------------------------+       *
C  NONSTANDARD FEATURES:   NONE DETECTED                               *
C*******************   END OF DOCUMENTATION PACKAGE   ******************
C***********************************************************************
C*****************  ITEMS DELETED FROM DOCPACK  ************************
C  =============================== DATE: 97/02/13 ==================   *
C NAME: FFT2DI    FFT - 2D, INVERSE                    REV 1.0  NOV 92 *
C********************  END OF DELETED ITEMS  ***************************
C***********************************************************************
      SUBROUTINE FFT2DI (MT, NT, NX, NK, NW, IW1, FILTER, DATAIN,
     &                   DATAOUT)
C
C  SUBROUTINE ARGUMENTS
C
      INTEGER MT, NT, NX, NK, NW, IW1
      REAL    FILTER(*), DATAOUT(*)
      COMPLEX DATAIN(*)
C
C  LOCAL VARIABLES
C
      INTEGER IFAC(19), IWRK, JD1, JD2, JI, JR, JR1, JR2, JW, NK1, NK2
      REAL    SCALE
C
C-----------------------------------------------------------------------
C
      CALL VCLR( DATAOUT, 1, MT*NK )
C
C===  FILTER, TRANSPOSE, AND PUT IN FFT ORDER
C
      JR  = 2 * IW1 - 1
      JI  = JR + 1
      NK1 = NK / 2 + 1
      NK2 = NK - NK1
      JR1 = JR
      JR2 = JR1 + NK1 * MT
      JD2 = 1
      JD1 = JD2 + NK2
      DO 110 JW = 1, NW
         SCALE = FILTER(JW)
         CALL CVSMUL (DATAIN(JD1), 2, SCALE, DATAOUT(JR1), MT, NK1)
         CALL CVSMUL (DATAIN(JD2), 2, SCALE, DATAOUT(JR2), MT, NK2)
         JR1 = JR1 + 2
         JR2 = JR2 + 2
         JD1 = JD1 + NK
         JD2 = JD2 + NK
  110 CONTINUE
C
C===  TRANSFORM FROM K TO X
C===  NOTE: DATAIN IS USED FOR WORK SPACE AND TRIG TABLES
C
      IWRK = 1 + NK
      CALL CFTFAX (NK, IFAC, DATAIN)
      CALL CFFTMLT (DATAOUT(JR), DATAOUT(JI), DATAIN(IWRK), DATAIN,
     &              IFAC, MT, 2, NK, NW, 1)
C
C===  TRANSFORM FROM OMEGA TO T
C===  NOTE: DATAIN IS USED FOR WORK SPACE AND TRIG TABLES
C
      IWRK = 1 + NT
      CALL FFTFAX (NT, IFAC, DATAIN)
      CALL RFFTMLT (DATAOUT, DATAIN(IWRK), DATAIN, IFAC, 1,
     &              MT, NT, NX, 1)
C
      RETURN
      END
