C***********************************************************************
C                                                                      *
c                 copyright 2001, Amoco Production Company             *
c                             All Rights Reserved                      *
c                     an affiliate of BP America Inc.                  *
C***********************************************************************
C***********************************************************************
C                                                                      *
C  ROUTINE:       FXCONT                                               *
C  ROUTINE TYPE:  SUBROUTINE                                           *
C  PURPOSE:                                                            *
C       PERFORMS A DOWNWARD CONTINUATION OF THE WAVE FIELD, PSI(X,Z),  *
C       FROM Z = Z0 TO Z = Z1 = Z0 + NZ * DELTAZ.  THE FIELD IS        *
C       CONTINUED FOR NZ STEPS OF DELTAZ EACH.                         *
C                                                                      *
C  ENTRY POINTS:                                                       *
C      FXCONT  (GTR,GTI,PSIR,PSII,NA,NX,NZ,NSKIP,LX,WPR,WPI,WNR,WNI)   *
C  ARGUMENTS:                                                          *
C      GTR     REAL     ??IOU*  (LX,0:NA-1) -                          *
C      GTI     REAL     ??IOU*  (LX,0:NA-1) -                          *
C      PSIR    REAL     ??IOU*  (NX)        -                          *
C      PSII    REAL     ??IOU*  (NX)        -                          *
C      NA      INTEGER  ??IOU*              -                          *
C      NX      INTEGER  ??IOU*              -                          *
C      NZ      INTEGER  ??IOU*              -                          *
C      NSKIP   INTEGER  ??IOU*              -                          *
C      LX      INTEGER  ??IOU*              -                          *
C      WPR     REAL     ??IOU*  (*)         -                          *
C      WPI     REAL     ??IOU*  (*)         -                          *
C      WNR     REAL     ??IOU*  (*)         -                          *
C      WNI     REAL     ??IOU*  (*)         -                          *
C       +------------------------------------------------------+       *
C       |               DEVELOPMENT INFORMATION                |       *
C       +------------------------------------------------------+       *
C  AUTHOR:   ???                                ORIGIN DATE: 97/02/13  *
C  LANGUAGE: FORTRAN 77                  DATE LAST COMPILED: 97/02/18  *
C       OTHERS: FORTRAN 77                                             *
C                                                                      *
C  CALLING FORMAT:                                                     *
C       CALL FXCONT( GTR, GTI, PSIR, PSII, NA, NX, NZ, NSKIP, LX,      *
C                    WPR, WPI, WNR, WNI )                              *
C                                                                      *
C  HISTORY:                                                            *
C       ORIGINAL        JAN 93          R.D. COLEMAN, CETech           *
C                                                                      *
C  PARAMETERS:                                                         *
C       GTR     REAL INPUT MATRIX OF DIMENSION LX BY NA                *
C               REAL COMPONENTS OF GREEN'S FUNCTION MATRIX.            *
C                                                                      *
C       GTI     REAL INPUT MATRIX OF DIMENSION LX BY NA                *
C               IMAGINARY COMPONENTS OF GREEN'S FUNCTION MATRIX.       *
C                                                                      *
C       PSIR    REAL INPUT/OUTPUT VECTOR OF LENGTH NX                  *
C               REAL COMPONENTS OF THE WAVE FIELD.                     *
C               ON INPUT, THE CONTENTS ARE FOR Z = Z0.  ON OUTPUT,     *
C               THE CONTENTS ARE FOR Z = Z1.                           *
C                                                                      *
C       PSII    REAL INPUT/OUTPUT VECTOR OF LENGTH NX                  *
C               IMAGINARY COMPONENTS OF THE WAVE FIELD.                *
C               ON INPUT, THE CONTENTS ARE FOR Z = Z0.  ON OUTPUT,     *
C               THE CONTENTS ARE FOR Z = Z1.                           *
C                                                                      *
C       NA      INTEGER INPUT SCALAR                                   *
C               NUMBER OF APERTURES.                                   *
C                                                                      *
C       NX      INTEGER INPUT SCALAR                                   *
C               NUMBER OF X'S.                                         *
C                                                                      *
C       NZ      INTEGER INPUT SCALAR                                   *
C               NUMBER OF Z'S.                                         *
C                                                                      *
C       NSKIP   INTEGER INPUT SCALAR                                   *
C               SKIP VALUE.                                            *
C                                                                      *
C       LX      INTEGER INPUT SCALAR                                   *
C               ROW DIMENSION OF GTR AND GTI.  LX MUST BE GREATER THAN *
C               OR EQUAL TO NX.                                        *
C                                                                      *
C       WPR     REAL SCRATCH VECTOR OF LENGTH NX+(NA-1)*NSKIP          *
C                                                                      *
C       WPI     REAL SCRATCH VECTOR OF LENGTH NX+(NA-1)*NSKIP          *
C                                                                      *
C       WNR     REAL SCRATCH VECTOR OF LENGTH NX+(NA-1)*NSKIP          *
C                                                                      *
C       WNI     REAL SCRATCH VECTOR OF LENGTH NX+(NA-1)*NSKIP          *
C                                                                      *
C  DESCRIPTION:                                                        *
C       THE DOWNWARD CONTINUATION OPERATION IS PERFORMED NZ TIMES.  THE*
C       OPERATION IS DEFINED BY                                        *
C                                                                      *
C       TEMP(I) := SUM[ F(I,J), J = 1, NA ], I = 1, NX                 *
C       PSI(I)  := TEMP(I), I = 1, NX                                  *
C                                                                      *
C       WHERE:                                                         *
C          F(I,J) = G(I,J) + H(I,J)                                    *
C          G(I,J) = PSI(I+(J-1)*NSKIP) * GT(I+(J-1)*NSKIP,J),          *
C                      IF  I+(J-1)*NSKIP <= NX                         *
C                 = 0.0, OTHERWISE                                     *
C          H(I,J) = PSI(I-(J-1)*NSKIP) * GT(I-(J-1)*NSKIP,J),          *
C                      IF I-(J-1)*NSKIP >= 1                           *
C                 = 0.0, OTHERWISE                                     *
C                                                                      *
C       NOTE: PSI(I)  = CMPLX( PSIR(I), PSII(I) )                      *
C             GT(I,J) = CMPLX( GTR(I,J), GTI(I,J) )                    *
C                                                                      *
C  SUBPROGRAMS CALLED:                                                 *
C       NONE                                                           *
C                                                                      *
C  ERROR CONDITIONS:                                                   *
C       NONE                                                           *
C                                                                      *
C--------------------------------------------------------------------- *
C                                                                      *
C       +------------------------------------------------------+       *
C       |                 EXTERNAL ENVIRONMENT                 |       *
C       +------------------------------------------------------+       *
C  ROUTINES CALLED:  NONE                                              *
C  INTRINSIC FUNCTIONS CALLED:  NONE                                   *
C  FILES:            NONE                                              *
C  COMMON:           NONE                                              *
C  STOP CODES:       NONE                                              *
C       +------------------------------------------------------+       *
C       |             OTHER DOCUMENTATION DETAILS              |       *
C       +------------------------------------------------------+       *
C  ERROR HANDLING:  ???                                                *
C  GENERAL DESCRIPTION:  ???                                           *
C  REVISED BY:  ???                           REVISION DATE: ?Y/?M/?D  *
C       +------------------------------------------------------+       *
C       |                 ANALYSIS INFORMATION                 |       *
C       +------------------------------------------------------+       *
C  NONSTANDARD FEATURES:   NONE DETECTED                               *
C*******************   END OF DOCUMENTATION PACKAGE   ******************
C***********************************************************************
C*****************  ITEMS DELETED FROM DOCPACK  ************************
C  =============================== DATE: 97/02/13 ==================   *
C NAME: FXCONT    FX DOWNWARD CONTINUATION             REL 1.0  JAN 93 *
C********************  END OF DELETED ITEMS  ***************************
C***********************************************************************
      SUBROUTINE FXCONT( GTR, GTI, PSIR, PSII, NA, NX, NZ, NSKIP, LX,
     &                   WPR, WPI, WNR, WNI )
C
#ifdef CRAY
      IMPLICIT NONE
#endif
C
      INTEGER MAXNX, MAXOFF
      PARAMETER( MAXNX = 2048, MAXOFF = 512 )
C
      INTEGER NA, NX, NZ, NSKIP, LX
      REAL    PSIR(NX), PSII(NX), GTR(LX,0:NA-1), GTI(LX,0:NA-1),
     &        WPR(*), WPI(*), WNR(*), WNI(*)
C
      INTEGER NOFF, JA, JX, JZ, JN, JP, JNOFF, JPOFF, JAOFF, JSEG,
     &        KX, MX, NSEG, NSEG1, NSEG2, NXSEG1, NXSEG2
      REAL    TR, TI
C
C-------------------------------------------------------------------
C
      IF( NSKIP .LE. 0 ) NSKIP = 1
      NOFF = NSKIP * ( NA - 1 )
C
#ifdef CRAY2
      NSEG   = ( NX + MAXNX - 1 ) / MAXNX
      NXSEG1 = NX / NSEG
C
C===  IF IT FITS, JUST CALL CONT
C
      IF( NX .LE. MAXNX .AND. NOFF .LE. MAXOFF ) THEN
         CALL CONT( GTR, GTI, PSIR, PSII, NA, NX, NZ, NSKIP, LX )
         RETURN
C
C===  ELSE SEGMENT AND USE CONT UNLESS NOFF IS TOO BIG
C
      ELSE IF( NOFF .LE. NXSEG1 .AND. NOFF .LE. MAXOFF ) THEN
         NXSEG2 = NXSEG1 + 1
         NSEG2  = NX - NSEG * NXSEG1
         NSEG1  = NSEG - NSEG2
C
C======  LOOP OVER Z
C
         DO 60 JZ = 1, NZ
C
C=========  SAVE A COPY OF PSI
C
            CALL VMOV( PSIR, 1, WPR, 1, NX )
            CALL VMOV( PSII, 1, WPI, 1, NX )
C
C=========  LOOP OVER SEGMENTS - CONTINUE EACH SEGMENT WITH CONT
C
            JX = 1
            MX = NXSEG1
            DO 10 JSEG = 1, NSEG
               IF( JSEG .GT. NSEG1 ) MX = NXSEG2
               CALL CONT( GTR(JX,0), GTI(JX,0), PSIR(JX), PSII(JX),
     &                    NA, MX, 1, NSKIP, LX )
               JX = JX + MX
   10       CONTINUE
C
C=========  LOOP OVER SEGMENTS - ADD CONTRIBUTION OF OVERLAP
C
            KX = 0
            MX = NXSEG1
            DO 50 JSEG = 1, NSEG-1
               IF( JSEG .GT. NSEG1 ) MX = NXSEG2
               KX = KX + MX
C
C============  LOOP OVER APERTURES
C
               DO 40 JA = 1, NA-1
                  JAOFF = JA * NSKIP
C
C===============  ADD OVERLAP TO THE LEFT
C
                  DO 20 JX = KX+1, KX+JAOFF
                     TR = WPR(JX) * GTR(JX,JA) - WPI(JX) * GTI(JX,JA)
                     TI = WPI(JX) * GTR(JX,JA) + WPR(JX) * GTI(JX,JA)
                     PSIR(JX-JAOFF) = PSIR(JX-JAOFF) + TR
                     PSII(JX-JAOFF) = PSII(JX-JAOFF) + TI
   20             CONTINUE
C
C===============  ADD OVERLAP TO THE RIGHT
C
                  DO 30 JX = KX+1-JAOFF, KX
                     TR = WPR(JX) * GTR(JX,JA) - WPI(JX) * GTI(JX,JA)
                     TI = WPI(JX) * GTR(JX,JA) + WPR(JX) * GTI(JX,JA)
                     PSIR(JX+JAOFF) = PSIR(JX+JAOFF) + TR
                     PSII(JX+JAOFF) = PSII(JX+JAOFF) + TI
   30             CONTINUE
   40          CONTINUE
   50       CONTINUE
   60    CONTINUE
         RETURN
      ENDIF
#endif
C
CCC   DELTA Z STEP LOOP
C
      DO 410 JZ = 1, NZ
C
CCC      INITIALIZE WORK ARRAY
C
         DO 110 JX = 1, NX+NOFF
            WPR(JX) = 0.0
            WPI(JX) = 0.0
            WNR(JX) = 0.0
            WNI(JX) = 0.0
  110    CONTINUE
C
CCC      PROCESS CONTRIBUTION FROM EACH PSI IN SEQUENCE
C
         DO 220 JA = 0, NA-1
            JPOFF = JA * NSKIP
            JNOFF = NOFF - JA * NSKIP
C
CDIR$ IVDEP
            DO 210 JX = 1, NX
               TR = PSIR(JX) * GTR(JX,JA) - PSII(JX) * GTI(JX,JA)
               TI = PSII(JX) * GTR(JX,JA) + PSIR(JX) * GTI(JX,JA)
               JP = JX + JPOFF
               JN = JX + JNOFF
               WPR(JP) = WPR(JP) + TR
               WPI(JP) = WPI(JP) + TI
               WNR(JN) = WNR(JN) + TR
               WNI(JN) = WNI(JN) + TI
  210       CONTINUE
  220    CONTINUE
C
CCC      SUM RESULTS INTO PSI FROM WORK
C
         DO 310 JX = 1, NX
            PSIR(JX) = WPR(JX) + WNR(JX+NOFF)
            PSII(JX) = WPI(JX) + WNI(JX+NOFF)
  310    CONTINUE
C
  410 CONTINUE
C
      RETURN
      END
