C***********************************************************************
C                                                                      *
c                 copyright 2001, Amoco Production Company             *
c                             All Rights Reserved                      *
c                     an affiliate of BP America Inc.                  *
C***********************************************************************
C***********************************************************************
C                                                                      *
C  ROUTINE:       GCFTLP                                               *
C  ROUTINE TYPE:  SUBROUTINE                                           *
C  PURPOSE:                                                            *
C       GENERATES A PACKED, COMPLEX VALUED FUNCTION LOOK-UP TABLE FOR  *
C       USE WITH XCFTLP OR CCFTLP.                                     *
C                                                                      *
C  ENTRY POINTS:                                                       *
C      GCFTLP  (ZR,ZI,IZ,N,XMIN,XMAX,IFFLG,TABLE,IERR)                 *
C  ARGUMENTS:                                                          *
C      ZR      REAL     ??IOU*  (*) -                                  *
C      ZI      REAL     ??IOU*  (*) -                                  *
C      IZ      INTEGER  ??IOU*      -                                  *
C      N       INTEGER  ??IOU*      -                                  *
C      XMIN    REAL     ??IOU*      -                                  *
C      XMAX    REAL     ??IOU*      -                                  *
C      IFFLG   INTEGER  ??IOU*      -                                  *
C      TABLE   REAL*8   ??IOU*  (*) -                                  *
C      IERR    INTEGER  ??IOU*      -                                  *
C       +------------------------------------------------------+       *
C       |               DEVELOPMENT INFORMATION                |       *
C       +------------------------------------------------------+       *
C  AUTHOR:   ???                                ORIGIN DATE: 92/07/22  *
C  LANGUAGE: FORTRAN 77                  DATE LAST COMPILED: 97/02/18  *
C        1  FORTRAN 77 (CFT77)                                         *
C                                                                      *
C  HISTORY:                                                            *
C       ORIGINAL        MAR 90          R.D. COLEMAN, QTC              *
C                                                                      *
C  CALLING FORMAT:                                                     *
C       CALL GCFTLP (ZR, ZI, IZ, N, XMIN, XMAX, IFFLG, TABLE, IERR)    *
C                                                                      *
C  PARAMETERS:                                                         *
C       ZR      REAL INPUT VECTOR OF LENGTH N                          *
C               REAL COMPONENT OF FUNCTION IF IFFLG = 0.               *
C               NOT USED IF IFFLG != 0.                                *
C                                                                      *
C       ZI      REAL INPUT VECTOR OF LENGTH N                          *
C               IMAGINARY COMPONENT OF FUNCTION IF IFFLG = 0.          *
C               NOT USED IF IFFLG != 0.                                *
C                                                                      *
C       IZ      INTEGER INPUT SCALAR                                   *
C               STRIDE OF VECTORS ZR AND ZI                            *
C                                                                      *
C       N       INTEGER INPUT SCALAR                                   *
C               NUMBER OF FUNCTION VALUES (LENGTH OF VECTORS ZR AND ZI)*
C               N MUST BE >= 2.                                        *
C                                                                      *
C       XMIN    REAL INPUT SCALAR                                      *
C               MINIMUM VALUE OF REAL FUNCTION ARGUMENT.               *
C                                                                      *
C       XMAX    REAL INPUT SCALAR                                      *
C               MAXIMUM VALUE OF REAL FUNCTION ARGUMENT.               *
C               XMAX MUST BE > XMIN                                    *
C                                                                      *
C       IFFLG   INTEGER INPUT SCALAR                                   *
C               FUNCTION FLAG, VALUES ARE:                             *
C                  0 - FUNCTION VALUES CONTAINED IN ZR AND ZI          *
C                  1 - Z = CEXP( i*X) = COS(X) + i*SIN(X)              *
C                 -1 - Z = CEXP(-i*X) = COS(X) - i*SIN(X)              *
C                  2 - Z = CEXP( i*SQRT(X))                            *
C                 -2 - Z = CEXP(-i*SQRT(X))                            *
C                                                                      *
C       TABLE   WORD OUTPUT VECTOR OF LENGTH NTAB = N+5                *
C               TABLE OF PACKED VALUES AND PARAMETERS:                 *
C                 WORD  FORMAT   CONTENTS                              *
C                    1  INTEGER  NTAB   - TOTAL LENGTH OF TABLE        *
C                    2  REAL     SCALE  - INDEX SCALE FACTOR           *
C                    3  REAL     OFFSET - INDEX OFFSET VALUE           *
C                    4  REAL     XMIN   - MINIMUM VALID ARGUMENT       *
C                    5  REAL     XMAX   - MAXIMUM VALID ARGUMENT       *
C               6:NTAB  PACKED   COMPLEX FUNCTION VALUES               *
C                                                                      *
C       IERR    INTEGER OUTPUT SCALAR                                  *
C               COMPLETION CODE.  VALUES ARE:                          *
C                  0 - NORMAL COMPLETION                               *
C                  1 - INVALID VALUE OF N OR XMAX                      *
C                  2 - INVALID VALUE OF IFFLG                          *
C                 21 - IFFLG = 2 OR -2 AND XMIN < 0.0                  *
C                                                                      *
C  DESCRIPTION:                                                        *
C       THIS ROUTINE GENERATES A COMPLEX VALUED FUNCTION LOOK-UP TABLE *
C       IN THE FORMAT REQUIRED BY THE ROUTINES XCFTLP AND CCFTLP.  THE *
C       FUNCTION IS OF THE FORM Z = F(X), WHERE X IS REAL AND Z IS     *
C       COMPLEX.  THE TABLE GENERATED CONTAINS N VALUES OF THE FUNCTION*
C       CORRESPONDING TO N EQUALLY SPACED VALUES OF X BETWEEN XMIN AND *
C       XMAX INCLUSELY.  IF IFFLG = 0, THEN THE REAL AND IMAGINARY     *
C       COMPONENTS OF THE FUNCTION, Z, ARE CONTAINED IN ZR AND ZI,     *
C       RESPECTIVELY, ON INPUT.  IF IFFLG != 0, THEN IFFLG SELECTS ONE *
C       OF SEVERAL PREDETERMINED FUNCTIONS (SEE ABOVE) TO BE COMPUTED  *
C       BY THIS ROUTINE.  IN THIS CASE, ZR, ZI, AND IZ ARE IGNORED AND *
C       NEVER ACCESSED. THE FUNCTION CORRESPONDING TO IFFLG = -K IS THE*
C       COMPLEX CONJUGATE OF THE FUNCTION CORRESPONDING TO IFFLG = K.  *
C                                                                      *
C       THE REAL AND IMAGINARY COMPONENTS OF EACH COMPLEX VALUE ARE    *
C       PACKED INTO A SINGLE 64-BIT WORD.  THE REAL COMPONENT IS       *
C       CONTAINED IN THE MOST SIGNIFICANT 32 BITS AND THE IMAGINARY    *
C       COMPONENT IS CONTAINED IN THE LEAST SIGNIFICANT 32 BITS.  THE  *
C       FORMAT OF EACH COMPONENT IS THE SAME AS THE  1  REAL FORMAT    *
C       EXCEPT THAT THE MANTISSA IS 16 BITS INSTEAD OF 48 BITS.        *
C                                                                      *
C  SUBPROGRAMS CALLED:                                                 *
C       CFT77 INTRINSICS: AND, OR, SHIFTR, SHIFTL                      *
C                                                                      *
C  ERROR CONDITIONS:                                                   *
C       IF ANY PARAMETER VALUE OR COMBINATION OF PARAMETER VALUES IS   *
C       INVALID (SEE ABOVE), THEN THE APPROPRIATE COMPLETION CODE IS   *
C       SET (SEE ABOVE) AND THE ROUTINE IS ABORTED.                    *
C                                                                      *
C----------------------------------------------------------------------*
C                                                                      *
C  IMPLEMENTATION NOTES:                                               *
C       1. DIFFERENT LOOPS WERE IMPLEMENTED FOR EACH FUNCTION TO       *
C          AVOID CONDITIONAL STATEMENTS IN THE LOOPS AND, THUS,        *
C          INSURE VECTORIZATION BY THE COMPILER.                       *
C       2. THE 64-BIT REALS WERE CONVERTED TO 32-BITS WITH ROUNDING.   *
C          THE METHOD USED WAS CHOOSEN TO AVOID CONDITIONAL STATEMENTS *
C          AS FOLLOWS:                                                 *
C             (A) FORM T, A 32-BIT TRUNCATED REAL                      *
C             (B) FORM B, AN UNNORMALIZED REAL WHOSE SIGN AND EXPONENT *
C                 ARE THE SAME AS THE ORIGINAL, Z, AND WHOSE MANTISSA  *
C                 IS ALL ZEROS EXCEPT FOR BIT 32 WHICH IS SET EQUAL TO *
C                 BIT 31 OF Z.                                         *
C             (C) ADD T AND B                                          *
C                                                                      *
C             THAT IS,                                                 *
C                                                                      *
C             (A) T = Z & 0xFFFFFFFF00000000                           *
C             (B) B = Z & 0x0000000080000000                           *
C                 B = B << 1                                           *
C                 E = Z & 0xFFFF000000000000                           *
C                 B = E | B                                            *
C             (C) T = T + B                                            *
C                                                                      *
C----------------------------------------------------------------------*
C                                                                      *
C       +------------------------------------------------------+       *
C       |                 EXTERNAL ENVIRONMENT                 |       *
C       +------------------------------------------------------+       *
C  ROUTINES CALLED:  NONE                                              *
C  INTRINSIC FUNCTIONS CALLED:                                         *
C      SIN     GENERIC -                                               *
C      IABS    INTEGER -                                               *
C      COS     GENERIC -                                               *
C      FLOAT   REAL    -                                               *
C      SQRT    GENERIC -                                               *
C  FILES:            NONE                                              *
C  COMMON:           NONE                                              *
C  STOP CODES:       NONE                                              *
C       +------------------------------------------------------+       *
C       |             OTHER DOCUMENTATION DETAILS              |       *
C       +------------------------------------------------------+       *
C  ERROR HANDLING:  ???                                                *
C  GENERAL DESCRIPTION:  ???                                           *
C  REVISED BY:  ???                           REVISION DATE: ?Y/?M/?D  *
C       +------------------------------------------------------+       *
C       |                 ANALYSIS INFORMATION                 |       *
C       +------------------------------------------------------+       *
C  NONSTANDARD FEATURES:      1 DETECTED                               *
C      REAL*                                                           *
C*******************   END OF DOCUMENTATION PACKAGE   ******************
C***********************************************************************
C*****************  ITEMS DELETED FROM DOCPACK  ************************
C  =============================== DATE: 92/07/22 ==================   *
C NAME: GCFTLP    GENERATE COMPLEX FUNCTION TABLE      REV 1.0  MAR 90 *
C  =============================== DATE: 97/02/13 ==================   *
C      TABLE   REAL     ??IOU*  (*) -                                  *
C      AND     REAL -                                                  *
C      SHIFTL  REAL -                                                  *
C      OR      REAL -                                                  *
C      SHIFTR  REAL -                                                  *
C  NONSTANDARD FEATURES:   NONE DETECTED                               *
C********************  END OF DELETED ITEMS  ***************************
C***********************************************************************
      SUBROUTINE GCFTLP (ZR, ZI, IZ, N, XMIN, XMAX, IFFLG, TABLE, IERR)
C
C  PARAMETERS:
C
      INTEGER IZ, N, IFFLG, IERR
      REAL    ZR(*), ZI(*), XMIN, XMAX
#ifdef CRAY
      REAL    TABLE(*)
#else
      REAL*8  TABLE(*)
#endif
C
C  LOCAL VARIABLES:
C
      INTEGER NTAB, J, JT, JZ, ISW, MAXFUN
      REAL    DELTA, SCALE, OFFSET, SGN, XJ, ZRJ, ZIJ
C
#ifdef CRAY
      INTEGER TMASK, EMASK, BMASK
      REAL    XNTAB, TR, TI, ER, EI, BR, BI
C
      EQUIVALENCE (NTAB, XNTAB)
C
      PARAMETER (TMASK = x'FFFFFFFF00000000')
      PARAMETER (EMASK = x'FFFF000000000000')
      PARAMETER (BMASK = x'0000000080000000')
C
      PARAMETER (MAXFUN = 2)
#else
      REAL    CVAL2(2)
      REAL*8  XNTAB, CVAL
C
      EQUIVALENCE (CVAL, CVAL2), (NTAB, XNTAB)
C
      PARAMETER (MAXFUN = 2)
C
      DATA XNTAB / 0.0D0 /
#endif
C
C-----------------------------------------------------------------------
C
      IF (N .LE. 1 .OR. XMIN .GE. XMAX) THEN
         IERR = 1
         RETURN
      ELSE IF (IFFLG .GT. MAXFUN .OR. IFFLG .LT. -MAXFUN) THEN
         IERR = 2
         RETURN
      ELSE IF (IABS( IFFLG ) .EQ. 2 .AND. XMIN .LT. 0.0) THEN
         IERR = 21
         RETURN
      ENDIF
C
      IERR = 0
C
      NTAB   = N + 5
      DELTA  = (XMAX - XMIN) / FLOAT( N - 1 )
      SCALE  = 1.0 / DELTA
      OFFSET = 6.5 - SCALE * XMIN
C
      TABLE(1) = XNTAB
      TABLE(2) = SCALE
      TABLE(3) = OFFSET
      TABLE(4) = XMIN
      TABLE(5) = XMAX
C
      IF (IFFLG .GE. 0) THEN
         SGN = 1.0
         ISW = IFFLG
      ELSE
         SGN = -1.0
         ISW = -IFFLG
      ENDIF
C
      GO TO (200, 300), ISW
C
      JT = 6
      JZ = 1
      DO 110 J = 1, N
         ZRJ = ZR(JZ)
         ZIJ = ZI(JZ)
C
#ifdef CRAY
         TR  = AND( ZRJ, TMASK )
         TI  = AND( ZIJ, TMASK )
         ER  = AND( ZRJ, EMASK )
         EI  = AND( ZIJ, EMASK )
         BR  = AND( ZRJ, BMASK )
         BI  = AND( ZIJ, BMASK )
         BR  = SHIFTL( BR, 1 )
         BI  = SHIFTL( BI, 1 )
         BR  = OR( BR, ER )
         BI  = OR( BI, EI )
         TR  = TR + BR
         TI  = TI + BI
         TI  = SHIFTR( TI, 32 )
         TABLE(JT) = OR( TR, TI )
#else
         CVAL2(1)  = ZRJ
         CVAL2(2)  = ZIJ
         TABLE(JT) = CVAL
#endif
C
         JT = JT + 1
         JZ = JZ + IZ
  110 CONTINUE
      RETURN
C
  200 CONTINUE
      JT = 6
      DO 210 J = 0, N-1
         XJ  = XMIN + DELTA * FLOAT( J )
         ZRJ = COS( XJ )
         ZIJ = SIN( XJ ) * SGN
C
#ifdef CRAY
         TR  = AND( ZRJ, TMASK )
         TI  = AND( ZIJ, TMASK )
         ER  = AND( ZRJ, EMASK )
         EI  = AND( ZIJ, EMASK )
         BR  = AND( ZRJ, BMASK )
         BI  = AND( ZIJ, BMASK )
         BR  = SHIFTL( BR, 1 )
         BI  = SHIFTL( BI, 1 )
         BR  = OR( BR, ER )
         BI  = OR( BI, EI )
         TR  = TR + BR
         TI  = TI + BI
         TI  = SHIFTR( TI, 32 )
         TABLE(JT) = OR( TR, TI )
#else
         CVAL2(1)  = ZRJ
         CVAL2(2)  = ZIJ
         TABLE(JT) = CVAL
#endif
C
         JT = JT + 1
  210 CONTINUE
      RETURN
C
  300 CONTINUE
      JT = 6
      DO 310 J = 0, N-1
         XJ  = SQRT( XMIN + DELTA * FLOAT( J ) )
         ZRJ = COS( XJ )
         ZIJ = SIN( XJ ) * SGN
C
#ifdef CRAY
         TR  = AND( ZRJ, TMASK )
         TI  = AND( ZIJ, TMASK )
         ER  = AND( ZRJ, EMASK )
         EI  = AND( ZIJ, EMASK )
         BR  = AND( ZRJ, BMASK )
         BI  = AND( ZIJ, BMASK )
         BR  = SHIFTL( BR, 1 )
         BI  = SHIFTL( BI, 1 )
         BR  = OR( BR, ER )
         BI  = OR( BI, EI )
         TR  = TR + BR
         TI  = TI + BI
         TI  = SHIFTR( TI, 32 )
         TABLE(JT) = OR( TR, TI )
#else
         CVAL2(1)  = ZRJ
         CVAL2(2)  = ZIJ
         TABLE(JT) = CVAL
#endif
C
         JT = JT + 1
  310 CONTINUE
      RETURN
C
      END
