C***********************************************************************
C                                                                      *
c                 copyright 2001, Amoco Production Company             *
c                             All Rights Reserved                      *
c                     an affiliate of BP America Inc.                  *
C***********************************************************************
C***********************************************************************
C                                                                      *
C  ROUTINE:       RDREC                                                *
C  ROUTINE TYPE:  SUBROUTINE                                           *
C  PURPOSE:                                                            *
C      RDREC READS ALL OF THE TRACES FOR AN ENTIRE SIS RECORD FROM THE *
C      INPUT DATA SET AND SELECTIVELY COPIES THE DATA TO THE DATA ARRAY*
C                                                                      *
C       FORTRAN 77                                                     *
C                                                                      *
C  HISTORY:                                                            *
C       ORIGINAL                NOV 91          R.D. COLEMAN, CETech   *
C       Revision 1.1            JUN 93          R.D. COLEMAN, CETech   *
C               Fixed bug for NX > NTRC.                               *
C                                                                      *
C  CALLING FORMAT:                                                     *
C       CALL RDREC( LUINP, LUERR, IREC, NTRC, ITRC1, ITRC2,            *
C      &            NSMP, ISMP1, ISMP2, NTOFF, NT, IT, NX, IX,         *
C      &            TRACE, HEADER, DATA, IERR )                        *
C                                                                      *
C  PARAMETERS:                                                         *
C       LUINP   INTEGER INPUT SCALAR                                   *
C               Logical unit number of input SIS dataset.              *
C                                                                      *
C       LUERR   INTEGER INPUT SCALAR                                   *
C               Logical unit number for error messages.  If LUERR < 0, *
C               then error messages are suppressed.                    *
C                                                                      *
C       IREC    INTEGER INPUT SCALAR                                   *
C               Record index of the input record (for use in error     *
C               messages only).                                        *
C                                                                      *
C       NTRC    INTEGER INPUT SCALAR                                   *
C               Number of traces in the input data set.                *
C                                                                      *
C       ITRC1   INTEGER INPUT SCALAR                                   *
C               Index of first trace to keep.                          *
C                                                                      *
C       ITRC2   INTEGER INPUT SCALAR                                   *
C               Index of last trace to keep (ITRC2 >= ITRC1).          *
C                                                                      *
C       NSMP    INTEGER INPUT SCALAR                                   *
C               Number of samples in the input data set.               *
C                                                                      *
C       ISMP1   INTEGER INPUT SCALAR                                   *
C               Index of first sample to keep.                         *
C                                                                      *
C       ISMP2   INTEGER INPUT SCALAR                                   *
C               Index of last sample to keep (ISMP2 >= ISMP1).         *
C                                                                      *
C       NTOFF   INTEGER INPUT SCALAR                                   *
C               Number of zero pad elements to place at the beginning o*
C               each (logical) column of DATA.                         *
C                                                                      *
C       NT      INTEGER INPUT SCALAR                                   *
C               Number of T's - logical row dimension of output matrix.*
C                                                                      *
C       IT      INTEGER INPUT SCALAR                                   *
C               Stride in the T dimension (i.e., between elements of th*
C               same column).                                          *
C                                                                      *
C       NX      INTEGER INPUT SCALAR                                   *
C               Number of X's - logical column dimension of output     *
C               matrix                                                 *
C                                                                      *
C       IX      INTEGER INPUT SCALAR                                   *
C               Stride in the X dimension (i.e., between elements of th*
C               same row).                                             *
C                                                                      *
C       TRACE   REAL SCRATCH VECTOR OF LENGTH NT+NWHDR                 *
C               Work space.  Note: NWHDR = 128 for Cray systems        *
C                                        =  64 otherwise               *
C                                                                      *
C       HEADER  INTEGER*2/INTEGER OUTPUT MATRIX OF DIMENSION 128 BY NX *
C               Trace header array.  Note: The array is INTEGER*2 on   *
C               non-Cray systems and INTEGER on Cray systems.          *
C                                                                      *
C       DATA    REAL OUTPUT MATRIX OF LOGICAL DIMENSION NT BY NX       *
C               Data array.  Note: The physical dimensions are a       *
C               function of the strides IT and IX.                     *
C                                                                      *
C       IERR    INTEGER OUTPUT SCALAR                                  *
C               Completion code.  IERR = 0 for normal completion       *
C                                     != 0 otherwise                   *
C                                                                      *
C  DESCRIPTION:                                                        *
C       RDREC reads all NTRC traces of an SIS record.  All data        *
C       (including trace headers) before trace ITRC1 and after trace   *
C       ITRC2 is ignored.  For the remaining traces, the trace header  *
C       is extracted and placed in the corresponding column of HEADER  *
C       and data samples ISMP1 through ISMP2 are extracted and placed i*
C       the corresponding logical column of DATA beginning at element  *
C       NTOFF+1.  The first NTOFF elements of each logical column of   *
C       DATA is set to zero.  Any elements in a logical column after   *
C       last data element extracted from the trace are also set to zero*
C       If NX > ITRC2-ITRC1+1, then the last NX-(ITRC2-ITRC1+1) columns*
C       of both DATA and HEADER are zero filled.                       *
C                                                                      *
C       The strides IX and IT allow the data contained in DATA to be   *
C       transposed and/or a submatrix.  Typically, IT = 1 and IX (>= NT*
C       is the row dimension of DATA.                                  *
C                                                                      *
C  SUBPROGRAMS CALLED:                                                 *
C       VCLR, VMOV, SKIPT, RTAPE                                       *
C                                                                      *
C  ERROR CONDITIONS:                                                   *
C       If an error is reported by RTAPE, then an error message is     *
C       written to LUERR, IERR is set to -1, and the routine is aborted*
C                                                                      *
C--------------------------------------------------------------------- *
C                                                                      *
C  ENTRY POINTS:                                                       *
C      RDREC  (LUINP,LUERR,IREC,NTRC,ITRC1,ITRC2,NSMP,ISMP1,ISMP2,     *
C              NTOFF,NT,IT,NX,IX,TRACE,HEADER,DATA,IERR)               *
C  ARGUMENTS:                                                          *
C      LUINP   INTEGER    ??IOU*             -                         *
C      LUERR   INTEGER    ??IOU*             -                         *
C      IREC    INTEGER    ??IOU*             -                         *
C      NTRC    INTEGER    ??IOU*             -                         *
C      ITRC1   INTEGER    ??IOU*             -                         *
C      ITRC2   INTEGER    ??IOU*             -                         *
C      NSMP    INTEGER    ??IOU*             -                         *
C      ISMP1   INTEGER    ??IOU*             -                         *
C      ISMP2   INTEGER    ??IOU*             -                         *
C      NTOFF   INTEGER    ??IOU*             -                         *
C      NT      INTEGER    ??IOU*             -                         *
C      IT      INTEGER    ??IOU*             -                         *
C      NX      INTEGER    ??IOU*             -                         *
C      IX      INTEGER    ??IOU*             -                         *
C      TRACE   REAL       ??IOU*  (*)        -                         *
C      HEADER  INTEGER*2  ??IOU*  (NTRHDR,*) -                         *
C      DATA    REAL       ??IOU*  (*)        -                         *
C      IERR    INTEGER    ??IOU*             -                         *
C       +------------------------------------------------------+       *
C       |               DEVELOPMENT INFORMATION                |       *
C       +------------------------------------------------------+       *
C  AUTHOR:   ???                                ORIGIN DATE: 97/02/13  *
C  LANGUAGE: FORTRAN 77                  DATE LAST COMPILED: 97/02/18  *
C       +------------------------------------------------------+       *
C       |                 EXTERNAL ENVIRONMENT                 |       *
C       +------------------------------------------------------+       *
C  ROUTINES CALLED:                                                    *
C      VCLR  -                                                         *
C      SKIPT -                                                         *
C      RTAPE -                                                         *
C      VMOV  -                                                         *
C  INTRINSIC FUNCTIONS CALLED:                                         *
C      MAX0    INTEGER -                                               *
C  FILES:                                                              *
C      LUERR  ( OUTPUT SEQUENTIAL ) -                                  *
C  COMMON:           NONE                                              *
C  STOP CODES:       NONE                                              *
C       +------------------------------------------------------+       *
C       |             OTHER DOCUMENTATION DETAILS              |       *
C       +------------------------------------------------------+       *
C  ERROR HANDLING:  ???                                                *
C  GENERAL DESCRIPTION:  ???                                           *
C  REVISED BY:  ???                           REVISION DATE: ?Y/?M/?D  *
C       +------------------------------------------------------+       *
C       |                 ANALYSIS INFORMATION                 |       *
C       +------------------------------------------------------+       *
C  NONSTANDARD FEATURES:      1 DETECTED                               *
C      INTEGER*                                                        *
C*******************   END OF DOCUMENTATION PACKAGE   ******************
C***********************************************************************
C*****************  ITEMS DELETED FROM DOCPACK  ************************
C  =============================== DATE: 97/02/13 ==================   *
C NAME: RDREC     READ SIS RECORD                      REV 1.1  JUN 93 *
C  LANGUAGE                                                            *
C********************  END OF DELETED ITEMS  ***************************
C***********************************************************************
      SUBROUTINE RDREC( LUINP, LUERR, IREC, NTRC, ITRC1, ITRC2,
     &                  NSMP, ISMP1, ISMP2, NTOFF, NT, IT, NX, IX,
     &                  TRACE, HEADER, DATA, IERR )
C
C  SYMBOLIC CONSTANTS
C
      INTEGER NTRHDR, NWBYTE, NHBYTE, NWHDR
      PARAMETER( NTRHDR = 128 )
#ifdef CRAY
      PARAMETER( NWBYTE = 8, NHBYTE = 8 )
#else
      PARAMETER( NWBYTE = 4, NHBYTE = 2 )
#endif
      PARAMETER( NWHDR = NTRHDR * NHBYTE / NWBYTE )
C
C  SUBROUTINE ARGUMENTS
C
      INTEGER   LUINP, LUERR, IREC, NTRC, ITRC1, ITRC2,
     &          NSMP, ISMP1, ISMP2, NTOFF, NT, IT, NX, IX, IERR
      REAL      DATA(*), TRACE(*)
#ifdef CRAY
      INTEGER   HEADER(NTRHDR,*)
#else
      INTEGER*2 HEADER(NTRHDR,*)
#endif
C
C  LOCAL VARIABLES
C
      INTEGER   IDATA, JH, JX, JD, MSMP, MTRC, JTRC, NBYTES
      REAL      AHEADR(NWHDR)
#ifdef CRAY
      INTEGER   IHEADR(NTRHDR)
#else
      INTEGER*2 IHEADR(NTRHDR)
#endif
C
      EQUIVALENCE ( IHEADR, AHEADR )
C
C-----------------------------------------------------------------------
C
  991 FORMAT (/' ', '***** TAPE READ ERROR IN RDREC - RECORD', I5,
     &              ' TRACE', I5, ' *****' /)
C
C-----------------------------------------------------------------------
C
      IDATA  = NWHDR + ISMP1
      MSMP   = ISMP2 - ISMP1 + 1
C
      CALL VCLR( DATA, 1, MAX0( NX*IX, NT*IT ) )
C
C===  IF ITRC1 > 1, SKIP THE FIRST ITRC1-1 TRACES
C
      MTRC = ITRC1 - 1
      IF( MTRC .GT. 0 ) THEN
         NBYTES = 4 * NSMP + 2 * NTRHDR
         CALL SKIPT( LUINP, MTRC, NBYTES )
      ENDIF
C
      JD   = 1 + NTOFF
      MTRC = ITRC2 - ITRC1 + 1
      DO 120 JX = 1, MTRC
         NBYTES = 0
         CALL RTAPE( LUINP, TRACE, NBYTES )
C
         IF (NBYTES .EQ. 0) THEN
            JTRC = JX + ITRC1 - 1
            IF( LUERR .GE. 0 ) WRITE (LUERR, 991) IREC, JTRC
            IERR = -1
            RETURN
         ENDIF
C
         CALL VMOV( TRACE, 1, AHEADR, 1, NWHDR )
         DO 110 JH = 1, NTRHDR
            HEADER(JH,JX) = IHEADR(JH)
  110    CONTINUE
C
         IF( HEADER(125,JX) .NE. 30000 ) THEN
            CALL VMOV( TRACE(IDATA), 1, DATA(JD), IT, MSMP )
         ENDIF
C
         JD = JD + IX
  120 CONTINUE
C
C===  IF NX > MTRC, THEN ZERO FILL LAST NX-MTRC COLUMNS OF HEADER
C
      IF( NX .GT. MTRC ) THEN
	 DO 220 JX = MTRC+1, NX
            DO 210 JH = 1, NTRHDR
               HEADER(JH,JX) = 0
  210       CONTINUE
  220    CONTINUE
      ENDIF
C
C===  IF ITRC2 < NTRC, SKIP THE LAST NTRC-ITRC2 TRACES
C
      MTRC = NTRC - ITRC2
      IF( MTRC .GT. 0 ) THEN
         NBYTES = 4 * NSMP + 2 * NTRHDR
         CALL SKIPT (LUINP, MTRC, NBYTES)
      ENDIF
C
      IERR = 0
      RETURN
      END
