C***********************************************************************
C                                                                      *
c                 copyright 2001, Amoco Production Company             *
c                             All Rights Reserved                      *
c                     an affiliate of BP America Inc.                  *
C***********************************************************************
C***********************************************************************
C                                                                      *
C  ROUTINE:       RFFTML                                               *
C  ROUTINE TYPE:  SUBROUTINE                                           *
C  PURPOSE:                                                            *
C       Computes multiple, real to complex forward or complex to real  *
C       inverse, mixed radix, in-place ffts.                           *
C       Functional equivalent of the Cray SCILIB routine of the same   *
C       name.                                                          *
C                                                                      *
C  ENTRY POINTS:                                                       *
C      RFFTMLT  (C,W,TRIGS,IFAX,ICE,ICV,NE,NV,IDIR)                    *
C  ARGUMENTS:                                                          *
C      C       REAL     ??IOU*  (*) -                                  *
C      W       REAL     ??IOU*  (*) -                                  *
C      TRIGS   REAL     ??IOU*  (1) -                                  *
C      IFAX    INTEGER  ??IOU*  (1) -                                  *
C      ICE     INTEGER  ??IOU*      -                                  *
C      ICV     INTEGER  ??IOU*      -                                  *
C      NE      INTEGER  ??IOU*      -                                  *
C      NV      INTEGER  ??IOU*      -                                  *
C      IDIR    INTEGER  ??IOU*      -                                  *
C       +------------------------------------------------------+       *
C       |               DEVELOPMENT INFORMATION                |       *
C       +------------------------------------------------------+       *
C  AUTHOR:   ???                                ORIGIN DATE: 97/02/13  *
C  LANGUAGE: FORTRAN 77                  DATE LAST COMPILED: 97/02/18  *
C       FORTRAN 77                                                     *
C                                                                      *
C  HISTORY:                                                            *
C       ORIGINAL                MAR 92          R.D. COLEMAN, CETech   *
C       REL 1.1                 NOV 94          R.D. COLEMAN, CETech   *
C               Added conditional code for Convex                      *
C                                                                      *
C  CALLING FORMAT:                                                     *
C       CALL RFFTMLT( C, W, TRIGS, IFAX, ICE, ICV, NE, NV, IDIR )      *
C                                                                      *
C  PARAMETERS:                                                         *
C       C       REAL/COMPLEX INPUT/OUTPUT MATRIX                       *
C               When IDIR = -1, then C contains NV real vectors of     *
C               length NE on input, and NV complex vectors of length   *
C               NE/2+1 on output.  The real components of the each     *
C               complex output vector replace the odd elements of each *
C               real input vector.  Likewise, the imaginary components *
C               replace the even elements of the real input vectors.   *
C               When IDIR = +1, the input and output data formats are  *
C               reversed.  The imaginary components of complex elements*
C               1 and NE/2+1 must be zero in each vector.              *
C                                                                      *
C       W       REAL SCRATCH VECTOR                                    *
C               For Cray systems, W is of length 2*NE*NV; otherwise,   *
C               W is of length 5*NE/2+18*NV+41.                        *
C                                                                      *
C       TRIGS   REAL INPUT VECTOR OF LENGTH 2*NE                       *
C               For Cray systems, TRIGS contains the SIN/COS table as  *
C               filled by SCILIB subroutine CFTFAX; otherwise, TRIGS is*
C               not used.                                              *
C                                                                      *
C       IFAX    INTEGER INPUT VECTOR OF LENGTH 19                      *
C               For Cray systems, IFAX contains the factors of NE as   *
C               filled by SCILIB subroutine CFTFAX; otherwise, IFAX is *
C               not used.                                              *
C                                                                      *
C       ICE     INTEGER INPUT SCALAR                                   *
C               Stride between real elements within a vector in C.     *
C                                                                      *
C       ICV     INTEGER INPUT SCALAR                                   *
C               Stride between the start of each vector in C.  For the *
C               best performance on Cray systems, ICV should be odd.   *
C                                                                      *
C       NE      INTEGER INPUT SCALAR                                   *
C               Number of elements in each vector - must be factorable *
C               as: NE = (2**p) * (3**q) * (5**r) where p is a positive*
C               integer and q and r are non-negative integers.         *
C                                                                      *
C       NV      INTEGER INPUT SCALAR                                   *
C               Number of vectors.                                     *
C                                                                      *
C       IDIR    INTEGER INPUT SCALAR                                   *
C               FFT direction switch: IDIR = -1 for forward FFT's and  *
C               IDIR = 1 for inverse FFT's.                            *
C                                                                      *
C  DESCRIPTION:                                                        *
C       If IDIR = -1, then RFFTMLT calculates:                         *
C                                                                      *
C          (c(1+2*k*ice+l*icv),c(1+2*k*ice+ice+l*icv)) =               *
C             SUM[ c(1+j*ice+l*icv) * exp( -2*pi*i*j*k/ne ),           *
C                  j = 0, ne-1 ] / ne                                  *
C             for k = 0, ne/2 and l = 0, nv-1                          *
C                                                                      *
C          where i = sqrt( -1 ).                                       *
C                                                                      *
C          Note that the results are scaled.                           *
C                                                                      *
C       If IDIR = +1, then RFFTMLT calculates:                         *
C                                                                      *
C          c(1+k*ice+l*icv) =                                          *
C             SUM[ (c(1+2*j*ice+l*icv),c(1+2*j*ice+ice+l*icv)) *       *
C                  exp( 2*pi*i*j*k/ne ), j = 0, ne-1 ]                 *
C             for k = 0, ne-1 and l = 0, nv-1                          *
C                                                                      *
C          where i = sqrt( -1 ).                                       *
C                                                                      *
C          Note that the complex elements for j > NE/2+1 are not stored*
C          but are obtained from the following relationship:           *
C             (c(1+2*j*ice+l*icv),c(1+2*j*ice+l*icv) =                 *
C                (c(1+2*(ne-j)*ice+l*icv),c(1+2*(ne-j)*ice+l*icv)      *
C                                                                      *
C       Vectorization is achieved by doing the transforms in parallel, *
C       with vector length = NV.                                       *
C                                                                      *
C  SUBPROGRAMS CALLED:                                                 *
C       Math Advantage: VCLR, VMOV, VSMUL                              *
C       Amoco Math Advantage Extensions: AMX006, AMX007                *
C                                                                      *
C  ERROR CONDITIONS:                                                   *
C       If any of the input parameters are invalid, then the results   *
C       are undefined.                                                 *
C                                                                      *
C----------------------------------------------------------------------*
C                                                                      *
C       +------------------------------------------------------+       *
C       |                 EXTERNAL ENVIRONMENT                 |       *
C       +------------------------------------------------------+       *
C  ROUTINES CALLED:                                                    *
C      AMX006 -                                                        *
C      VMOV   -                                                        *
C      AMX007 -                                                        *
C      VSMUL  -                                                        *
C      VCLR   -                                                        *
C  INTRINSIC FUNCTIONS CALLED:                                         *
C      FLOAT   REAL -                                                  *
C  FILES:            NONE                                              *
C  COMMON:           NONE                                              *
C  STOP CODES:       NONE                                              *
C       +------------------------------------------------------+       *
C       |             OTHER DOCUMENTATION DETAILS              |       *
C       +------------------------------------------------------+       *
C  ERROR HANDLING:  ???                                                *
C  GENERAL DESCRIPTION:  ???                                           *
C  REVISED BY:  ???                           REVISION DATE: ?Y/?M/?D  *
C       +------------------------------------------------------+       *
C       |                 ANALYSIS INFORMATION                 |       *
C       +------------------------------------------------------+       *
C  NONSTANDARD FEATURES:   NONE DETECTED                               *
C*******************   END OF DOCUMENTATION PACKAGE   ******************
C***********************************************************************
C*****************  ITEMS DELETED FROM DOCPACK  ************************
C  =============================== DATE: 97/02/13 ==================   *
C NAME: RFFTMLT   MULTIPLE REAL FFT - MIXED RADIX      REL 1.1  NOV 94 *
C********************  END OF DELETED ITEMS  ***************************
C***********************************************************************
      SUBROUTINE RFFTMLT( C, W, TRIGS, IFAX, ICE, ICV, NE, NV, IDIR )
C
C  PARAMETERS:
C
      REAL    C(*), W(*), TRIGS(1)
      INTEGER IFAX(1), ICE, ICV, NE, NV, IDIR
C
C  LOCAL VARIABLES:
C
#ifdef __convex__
      INTEGER IERR
C                      ERROR FLAG
#endif
      INTEGER J
C                      LOOP INDEX
      INTEGER JCON
C                      POINTER TO START OF CONSTANT TABLE IN W
      INTEGER JCOS
C                      POINTER TO START OF MAIN COSINE TABLE IN W
      INTEGER JCS2
C                      POINTER TO START OF SECONDARY COSINE TABLE IN W
      INTEGER JDES
C                      POINTER TO START OF DESTINATION TABLE IN W
      INTEGER JFAC
C                      POINTER TO START OF FACTOR TABLE IN W
      INTEGER JSIN
C                      POINTER TO START OF MAIN SINE TABLE IN W
      INTEGER JSN2
C                      POINTER TO START OF SECONDARY SINE TABLE IN W
      INTEGER JSRC
C                      POINTER TO START OF SOURCE TABLE IN W
      INTEGER JV
C                      POINTER TO VECTOR J
      INTEGER JWRK
C                      POINTER TO START OF 18*NV WORK SPACE IN W
      INTEGER KDIR
C                      INTERNAL DIRECTION FLAG
      INTEGER MCON
C                      SPACE RESERVED FOR CONSTANT TABLE IN W
      INTEGER MFAC
C                      SPACE RESERVED FOR FACTOR TABLE IN W
      INTEGER N2
C                      NE / 2
      INTEGER N4
C                      NE / 4
      INTEGER NF
C                      NUMBER OF FACTORS = ELEMENT COUNT OF FACTOR TABLE
      INTEGER NP
C                      NUMBER OF PERMUTATIONS = ELEMENT COUNT OF SOURCE
C                      AND DESTINATION TABLES
      REAL    S
C                      SCALE VALUE
 
C
      PARAMETER (MCON = 11, MFAC = 30)
C
C-----------------------------------------------------------------------
C
C  SET INTERNAL DIRECTION FLAG
C
      IF      (IDIR .GT. 0) THEN
         KDIR = -1
      ELSE IF (IDIR .LT. 0) THEN
         KDIR =  1
      ELSE
         RETURN
      ENDIF
 
#ifdef __convex__
      CALL SRCFTS( C, NE, ICE, NV, ICV, KDIR, IERR )
      IF( IERR .NE. 0 ) RETURN
 
      IF( KDIR .GT. 0 ) THEN
         S  = 1.0 / FLOAT( NE )
      ELSE
	 S  = FLOAT( NE )
      ENDIF
 
      JV = 1
      DO J = 1, NV
         CALL VSMUL( C(JV), ICE, S, C(JV), ICE, NE+1 )
         JV = JV + ICV
      ENDDO
#else
C
C  INITIALIZE TABLE POINTERS
C
      N2   = NE / 2
      N4   = NE / 4
      JCON = 1
      JSIN = JCON + MCON
      JCOS = JSIN + N2
      JSN2 = JCOS + N2
      JCS2 = JSN2 + N4
      JFAC = JCS2 + N4
      JSRC = JFAC + MFAC
      JDES = JSRC + N2
      JWRK = JDES + N2
C
C=======================================================================
C                           INITIALIZE TABLES
C=======================================================================
C
      CALL AMX006( ICE, NE, KDIR, W(JFAC), NF, W(JCON), W(JSIN),
     &             W(JCOS), W(JSN2), W(JCS2), W(JSRC), W(JDES), NP )
C
C  CHECK IF NE FACTORED OK
C
      IF( NF .EQ. 0 ) RETURN
C
C=======================================================================
C                              PERFORM FFT
C=======================================================================
C
      IF( KDIR .LT. 0 ) CALL VMOV( C(1+ICE*NE), ICV, C(1+ICE), ICV, NV )
C
      CALL AMX007( C, ICE, ICV, NE, NV, KDIR, W(JFAC), NF, W(JCON),
     &             W(JSIN), W(JCOS), W(JSN2), W(JCS2),
     &             W(JSRC), W(JDES), NP, W(JWRK) )
C
      IF( KDIR .GT. 0 ) THEN
         S  = 1.0 / FLOAT( 2 * NE )
         JV = 1
         DO 110 J = 1, NV
            CALL VSMUL( C(JV), ICE, S, C(JV), ICE, NE )
            JV = JV + ICV
  110    CONTINUE
         CALL VMOV( C(1+ICE), ICV, C(1+ICE*NE), ICV, NV )
         CALL VCLR( C(1+ICE*(NE+1)), ICV, NV )
         CALL VCLR( C(1+ICE)       , ICV, NV )
      ENDIF
C
C=======================================================================
C                               EXIT ROUTINE
C=======================================================================
C
#endif
      RETURN
      END
